/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.internal.processor;

import static com.mulesoft.modules.oauth2.provider.api.Constants.CLIENT_ID_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.REDIRECT_URI_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RESPONSE_TYPE_PARAMETER;
import static org.apache.commons.lang3.StringUtils.trimToEmpty;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

import com.mulesoft.modules.oauth2.provider.internal.OAuth2ProviderSecurityException;

/**
 * Error handling defined in:
 * http://tools.ietf.org/html/draft-ietf-oauth-v2-28#section-4.1.2.1
 */
public class RequestProcessingException extends OAuth2ProviderSecurityException {

  private static final long serialVersionUID = 7580399665743587412L;

  private final ErrorType errorType;

  public static enum ErrorType {
    /**
     * missing, invalid, or mismatching redirection URI
     */
    INVALID_REDIRECTION_URI(false, REDIRECT_URI_PARAMETER),
    /**
     * client identifier is missing or invalid
     */
    INVALID_CLIENT_ID(CLIENT_ID_PARAMETER),
    /**
     * Client authentication failed (e.g. unknown client, no client
     * authentication included, or unsupported authentication method)
     */
    INVALID_CLIENT,
    /**
     * The provided authorization grant (e.g. authorization code, resource owner
     * credentials) or refresh token is invalid, expired, revoked, does not match
     * the redirection URI used in the authorization request, or was issued to
     * another client.
     */
    INVALID_GRANT,
    /**
     * The request is missing a required parameter, includes an invalid parameter
     * value, includes a parameter more than once, or is otherwise malformed.
     */
    INVALID_REQUEST,
    /**
     * The client is not authorized to request an authorization code using this
     * method, or the authenticated client is not authorized to use this
     * authorization grant type.
     */
    UNAUTHORIZED_CLIENT,
    /**
     * The resource owner or authorization server denied the request.
     */
    ACCESS_DENIED,
    /**
     * The authorization server does not support obtaining an authorization code
     * using this method.
     */
    UNSUPPORTED_RESPONSE_TYPE(RESPONSE_TYPE_PARAMETER),
    /**
     * The authorization grant type is not supported by the authorization server.
     */
    UNSUPPORTED_GRANT_TYPE(),
    /**
     * The requested scope is invalid, unknown, or malformed.
     */
    INVALID_SCOPE,
    /**
     * The authorization server is currently unable to handle the request due to
     * a temporary overloading or maintenance of the server.
     */
    TEMPORARILY_UNAVAILABLE,
    /**
     * The authorization server encountered an unexpected condition which
     * prevented it from fulfilling the request.
     */
    SERVER_ERROR,
    /**
     * Requests that utilize the resource owner's password MUST be protected
     * against brute force attacks.
     */
    RATE_LIMIT_EXCEEDED(false);

    private final boolean doRedirect;
    private final String relatedParameter;

    private ErrorType() {
      this(true);
    }

    private ErrorType(final boolean doRedirect) {
      this(doRedirect, null);
    }

    private ErrorType(final String relatedParameter) {
      this(true, relatedParameter);
    }

    private ErrorType(final boolean doRedirect, final String relatedParameter) {
      this.doRedirect = doRedirect;
      this.relatedParameter = relatedParameter;
    }

    public String getErrorCode() {
      return toString().toLowerCase();
    }

    public boolean isDoRedirect() {
      return doRedirect;
    }

    public static ErrorType findByParameterName(final String parameterName) {
      checkArgument(parameterName != null, "parameterName can't be empty");

      for (final ErrorType et : values()) {
        if (parameterName.equals(et.relatedParameter)) {
          return et;
        }
      }

      return null;
    }
  };

  public RequestProcessingException(final ErrorType errorType) {
    this(errorType, null, null);
  }

  public RequestProcessingException(final ErrorType errorType, final String errorDescription) {
    this(errorType, errorDescription, null);
  }

  public RequestProcessingException(final ErrorType errorType, final Throwable cause) {
    this(errorType, null, cause);
  }

  public RequestProcessingException(final ErrorType errorType,
                                    final String errorDescription,
                                    final Throwable cause) {
    super(trimToEmpty(errorDescription), cause);
    this.errorType = errorType;
  }

  public ErrorType getErrorType() {
    return errorType;
  }
}
