/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider.internal.processor;

import static com.mulesoft.modules.oauth2.provider.api.Constants.ACCESS_TOKEN_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.CODE_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.EXPIRES_IN_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.HTTP_AUTHORIZATION_SCHEME_BASIC;
import static com.mulesoft.modules.oauth2.provider.api.Constants.REDIRECT_URI_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.REFRESH_TOKEN_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.AUTHORIZATION_CODE;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.CLIENT_CREDENTIALS;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.PASSWORD;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.REFRESH_TOKEN;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.TOKEN;
import static com.mulesoft.modules.oauth2.provider.api.Constants.SCOPE_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.TOKEN_TYPE_PARAMETER;
import static com.mulesoft.modules.oauth2.provider.api.client.ClientType.CONFIDENTIAL;
import static com.mulesoft.modules.oauth2.provider.internal.Utils.stringifyScopes;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.ACCESS_DENIED;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.INVALID_CLIENT;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.INVALID_CLIENT_ID;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.INVALID_REDIRECTION_URI;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.INVALID_SCOPE;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.UNAUTHORIZED_CLIENT;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingException.ErrorType.UNSUPPORTED_GRANT_TYPE;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingExceptionFactory.unkownClientIdException;
import static com.mulesoft.modules.oauth2.provider.internal.processor.RequestProcessingExceptionFactory.wrongClientSecretException;
import static org.apache.commons.collections.CollectionUtils.isNotEmpty;
import static org.apache.commons.collections.CollectionUtils.isSubCollection;
import static org.apache.commons.lang.StringEscapeUtils.escapeHtml;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.StringUtils.trimToNull;
import static org.mule.runtime.api.metadata.MediaType.APPLICATION_JSON;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.http.api.HttpHeaders.Names.AUTHORIZATION;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.runtime.http.api.HttpHeaders.Names.WWW_AUTHENTICATE;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.UNAUTHORIZED;
import org.mule.runtime.api.security.SecurityException;
import org.mule.runtime.http.api.domain.entity.ByteArrayHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponseBuilder;

import com.google.gson.Gson;
import com.mulesoft.modules.oauth2.provider.api.ResourceOwnerAuthentication;
import com.mulesoft.modules.oauth2.provider.api.client.Client;
import com.mulesoft.modules.oauth2.provider.api.AuthorizationRequest;
import com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType;
import com.mulesoft.modules.oauth2.provider.api.client.NoSuchClientException;
import com.mulesoft.modules.oauth2.provider.api.code.AuthorizationCodeStoreHolder;
import com.mulesoft.modules.oauth2.provider.internal.config.OAuthConfiguration;
import com.mulesoft.modules.oauth2.provider.api.token.Token;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

public class TokenRequestProcessor extends OAuth2ProviderRequestProcessor {

  private final static ResourceOwnerAuthentication NO_RESOURCE_OWNER_AUTHENTICATION = null;

  public TokenRequestProcessor(final OAuthConfiguration configuration) {
    super(configuration);
  }

  public void processRequest(final RequestData requestData, HttpResponseBuilder httpResponseBuilder)
      throws SecurityException {
    final RequestGrantType grantType = getSupportedRequestGrantTypeOrFail(requestData);

    final Client client = getKnownClientOrFail(requestData);
    if ((client.getType() == CONFIDENTIAL) && (!validateClientCredentials(client, requestData))) {
      throw wrongClientSecretException();
    }

    if (!client.isGrantTypeAuthorized(grantType)) {
      throw new RequestProcessingException(UNSUPPORTED_GRANT_TYPE, "Client does not support grant type: " + grantType);
    }

    if (grantType == AUTHORIZATION_CODE) {
      processAuthorizationCodeRequest(grantType, client, requestData, httpResponseBuilder);
    } else if (grantType == REFRESH_TOKEN) {
      processRefreshTokenRequest(grantType, client, requestData, httpResponseBuilder);
    } else if (grantType == PASSWORD) {
      processPasswordRequest(grantType, client, requestData, httpResponseBuilder);
    } else if (grantType == CLIENT_CREDENTIALS) {
      processClientCredentialsRequest(grantType, client, requestData, httpResponseBuilder);
    } else {
      throw new RequestProcessingException(UNSUPPORTED_GRANT_TYPE, "Unsupported grant type: "
          + grantType);
    }
  }

  @Override
  protected RequestProcessingException convertToRequestProcessingException(Exception e) {
    if (e instanceof NoSuchClientException) {
      return unkownClientIdException();
    }
    return super.convertToRequestProcessingException(e);
  }

  private void processAuthorizationCodeRequest(final RequestGrantType grantType,
                                               final Client client,
                                               final RequestData requestData,
                                               final HttpResponseBuilder httpResponseBuilder)
      throws SecurityException {
    final String authorizationCode = getMandatoryParameterOrFail(requestData, CODE_PARAMETER);
    final String redirectUri = getMandatoryParameterOrFail(requestData, REDIRECT_URI_PARAMETER);

    final AuthorizationCodeStoreHolder authorizationCodeStoreHolder = configuration.getAuthorizationCodeManager()
        .consumeAuthorizationCode(authorizationCode);

    final AuthorizationRequest pendingAuthorizationRequest = authorizationCodeStoreHolder.getAuthorizationRequest();
    if (!pendingAuthorizationRequest.getClientId().equals(client.getClientId())) {
      throw new RequestProcessingException(INVALID_CLIENT_ID);
    }

    if (!StringUtils.equals(trimToNull(redirectUri),
                            trimToNull(pendingAuthorizationRequest.getRedirectUri()))) {
      throw new RequestProcessingException(INVALID_REDIRECTION_URI);
    }

    final Token accessToken = configuration.getTokenManager().grantAccessToken(grantType,
                                                                               pendingAuthorizationRequest,
                                                                               NO_RESOURCE_OWNER_AUTHENTICATION);

    respondToken(accessToken, httpResponseBuilder);
  }

  private void processRefreshTokenRequest(final RequestGrantType grantType,
                                          final Client client,
                                          final RequestData requestData,
                                          final HttpResponseBuilder httpResponseBuilder)
      throws SecurityException {
    final String refreshToken = getMandatoryParameterOrFail(requestData, REFRESH_TOKEN_PARAMETER);
    final Set<String> effectiveScopes = getEffectiveScopes(requestData, client);

    final Token accessToken = configuration.getTokenManager().exchangeRefreshToken(refreshToken,
                                                                                   client.getClientId());

    if (isNotEmpty(effectiveScopes) && !isSubCollection(effectiveScopes, accessToken.getScopes())) {
      throw new RequestProcessingException(INVALID_SCOPE, "Scope doesn't match originally granted scope");
    }

    respondToken(accessToken, httpResponseBuilder);
  }

  private void processPasswordRequest(final RequestGrantType grantType,
                                      final Client client,
                                      final RequestData requestData,
                                      final HttpResponseBuilder httpResponseBuilder)
      throws SecurityException {
    final Pair<Boolean, ResourceOwnerAuthentication> resourceOwnerAuthenticationResult = validateResourceOwnerCredentials(
                                                                                                                          client,
                                                                                                                          requestData);
    if (!resourceOwnerAuthenticationResult.getLeft()) {
      throw new RequestProcessingException(ACCESS_DENIED);
    }

    final Set<String> effectiveScopes = getEffectiveScopes(requestData, client);

    final Token accessToken = configuration.getTokenManager().grantAccessToken(TOKEN,
                                                                               client.getClientId(), effectiveScopes,
                                                                               resourceOwnerAuthenticationResult.getRight());

    respondToken(accessToken, httpResponseBuilder);
  }

  private void processClientCredentialsRequest(final RequestGrantType grantType,
                                               final Client client,
                                               final RequestData requestData,
                                               final HttpResponseBuilder httpResponseBuilder)
      throws SecurityException {
    // the client credentials grant type MUST only be used by confidential
    // clients
    if (client.getType() != CONFIDENTIAL) {
      throw new RequestProcessingException(UNAUTHORIZED_CLIENT,
                                           "Client is not confidential!");
    }

    if (!validateClientCredentials(client, requestData)) {
      throw wrongClientSecretException();
    }

    final Set<String> effectiveScopes = getEffectiveScopes(requestData, client);

    // a refresh token SHOULD NOT be included
    final Token accessToken = configuration.getTokenManager().grantAccessToken(TOKEN,
                                                                               client.getClientId(), effectiveScopes,
                                                                               NO_RESOURCE_OWNER_AUTHENTICATION);

    respondToken(accessToken, httpResponseBuilder);
  }

  private void respondToken(final Token accessToken, final HttpResponseBuilder httpResponseBuilder) {
    final Map<String, Object> response = new HashMap<String, Object>();
    response.put(ACCESS_TOKEN_PARAMETER, accessToken.getAccessToken());
    response.put(TOKEN_TYPE_PARAMETER, accessToken.getType());
    response.put(EXPIRES_IN_PARAMETER, configuration.getTokenConfig().getTokenTtlInSeconds());

    if (isNotEmpty(accessToken.getScopes())) {
      // optional but doesn't hurt to output effective scope in case we
      // decide to support scope narrowing, see:
      // http://tools.ietf.org/html/draft-ietf-oauth-v2-28#section-5.1
      response.put(SCOPE_PARAMETER, stringifyScopes(accessToken.getScopes()));
    }

    if (isNotBlank(accessToken.getRefreshToken())) {
      response.put(REFRESH_TOKEN_PARAMETER, accessToken.getRefreshToken());
    }

    httpResponseBuilder.addHeader(CONTENT_TYPE, APPLICATION_JSON.toRfcString());
    httpResponseBuilder.entity(new ByteArrayHttpEntity(new Gson().toJson(response).getBytes()));

  }

  @Override
  protected void handleException(final RequestProcessingException exception,
                                 final RequestData requestData,
                                 final HttpResponseBuilder httpResponseBuilder) {

    super.handleException(exception, requestData, httpResponseBuilder);

    /*
     * see: http://tools.ietf.org/html/draft-ietf-oauth-v2-28#section-5.2
     */
    if (isNotBlank(getOptionalParameter(requestData, AUTHORIZATION))
        && (exception.getErrorType() == INVALID_CLIENT)) {
      httpResponseBuilder.statusCode(UNAUTHORIZED.getStatusCode());
      httpResponseBuilder.addHeader(WWW_AUTHENTICATE, HTTP_AUTHORIZATION_SCHEME_BASIC + " realm=\"OAuth2 Client Realm\"");
    }
  }

  @Override
  protected boolean isRedirectingForError(final RequestProcessingException.ErrorType errorType, final String redirectUri) {
    // never redirect in case of error
    return false;
  }


  @Override
  protected void setResponsePayload(HttpResponseBuilder responseBuilder, String encoding, String... parameters) {
    responseBuilder.addHeader(CONTENT_TYPE, APPLICATION_JSON.toRfcString());
    // will be serialized to json by a response transformer
    responseBuilder.entity(new ByteArrayHttpEntity(new Gson().toJson(keyValuePairsToMap((Object[]) parameters)).getBytes()));
  }

  @Override
  protected Map<String, Object> keyValuePairsToMap(final Object... parameters) {
    checkArgument(parameters.length % 2 == 0, "need an even number of (param name, param value) string pairs");
    final Map<String, Object> result = new HashMap<String, Object>();
    for (int i = 0; i < parameters.length; i += 2) {
      final Object value = parameters[i + 1];
      if (value == null) {
        continue;
      }
      final String key = (String) parameters[i];
      result.put(key, escapeHtml((String) value));
    }
    return result;
  }

}
