/*
 * (c) 2003-2019 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.configuration.properties.api;

import static com.mulesoft.modules.configuration.properties.api.EncryptionAlgorithm.AES;
import static com.mulesoft.modules.configuration.properties.api.EncryptionMode.CBC;
import static org.mule.encryption.jce.JCE.isJCEInstalled;
import static org.mule.runtime.api.component.ComponentIdentifier.builder;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.i18n.I18nMessageFactory;
import org.mule.runtime.api.util.Preconditions;
import org.mule.runtime.config.api.dsl.model.ConfigurationParameters;
import org.mule.runtime.config.api.dsl.model.ResourceProvider;
import org.mule.runtime.config.api.dsl.model.properties.ConfigurationPropertiesProviderFactory;

import com.mulesoft.modules.configuration.properties.internal.SecureConfigurationPropertiesProvider;

import java.util.List;

/**
 * Builds the provider for the secure-configuration-properties element.
 *
 * @since 1.0
 */
public class SecureConfigurationPropertiesProviderFactory implements ConfigurationPropertiesProviderFactory {

  public static final String EXTENSION_NAMESPACE = "secure-properties";
  public static final String SECURE_CONFIGURATION_PROPERTIES_ELEMENT = "config";
  public static final ComponentIdentifier SECURE_CONFIGURATION_PROPERTIES =
      builder().namespace(EXTENSION_NAMESPACE).name(SECURE_CONFIGURATION_PROPERTIES_ELEMENT).build();

  private static final String SHORT_KEY_MESSAGE = "You need to increment your key size." +
      " The minimum allowed key size is: %d, but your key size is: %d";
  private static final String LONG_KEY_MESSAGE = "Your key size exceeds the maximum allowed key size in your JVM." +
      " The maximum allowed key size is: %d, but your key size is: %d.";
  private static final String INSTALL_JCE_MESSAGE = "You need to install the Java Cryptography Extension (JCE) " +
      "Unlimited Strength Jurisdiction Policy Files";

  @Override
  public ComponentIdentifier getSupportedComponentIdentifier() {
    return SECURE_CONFIGURATION_PROPERTIES;
  }

  @Override
  public SecureConfigurationPropertiesProvider createProvider(ConfigurationParameters parameters,
                                                              ResourceProvider externalResourceProvider) {
    String file = parameters.getStringParameter("file");
    Preconditions.checkArgument(file != null, "Required attribute 'file' of 'secure-configuration-properties' not found");

    String key = parameters.getStringParameter("key");
    Preconditions.checkArgument(key != null, "Required attribute 'key' of 'secure-configuration-properties' not found");

    ComponentIdentifier encryptComponentIdentifier =
        ComponentIdentifier.builder().namespace(EXTENSION_NAMESPACE).name("encrypt").build();

    EncryptionAlgorithm algorithm = getAlgorithm(parameters.getComplexConfigurationParameter(encryptComponentIdentifier));
    EncryptionMode mode = getMode(parameters.getComplexConfigurationParameter(encryptComponentIdentifier));

    validateKeyLength(key, algorithm);

    return new SecureConfigurationPropertiesProvider(externalResourceProvider, file, algorithm, key, mode);
  }

  private EncryptionAlgorithm getAlgorithm(List<ConfigurationParameters> encryptionParameters) {
    return EncryptionAlgorithm.valueOf(getProperty(encryptionParameters, "algorithm", AES.name()));
  }

  private EncryptionMode getMode(List<ConfigurationParameters> encryptionParameters) {
    return EncryptionMode.valueOf(getProperty(encryptionParameters, "mode", CBC.name()));
  }

  private String getProperty(List<ConfigurationParameters> encryptionParameters, String property, String defaultValue) {
    if (encryptionParameters.size() != 1) {
      return defaultValue;
    }

    String propertyValue = encryptionParameters.get(0).getStringParameter(property);
    return propertyValue != null ? propertyValue : defaultValue;
  }

  private void validateKeyLength(String key, EncryptionAlgorithm algorithm) {
    if (key.length() > algorithm.getMaxKeySize()) {
      String message = String.format(LONG_KEY_MESSAGE, algorithm.getMaxKeySize(), key.length());
      if (!isJCEInstalled()) {
        message += INSTALL_JCE_MESSAGE;
      }

      throw new MuleRuntimeException(I18nMessageFactory.createStaticMessage(message));
    } else if (key.length() < algorithm.getMinKeySize()) {
      String message = String.format(SHORT_KEY_MESSAGE, algorithm.getMinKeySize(), key.length());
      throw new MuleRuntimeException(I18nMessageFactory.createStaticMessage(message));
    }
  }

}
