/*
 * (c) 2003-2019 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.configuration.properties.internal;

import static java.lang.String.format;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import org.mule.encryption.exception.MuleEncryptionException;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.config.api.dsl.model.ResourceProvider;
import org.mule.runtime.config.api.dsl.model.properties.ConfigurationProperty;
import org.mule.runtime.config.api.dsl.model.properties.DefaultConfigurationPropertiesProvider;
import org.mule.runtime.core.api.util.IOUtils;

import com.mulesoft.modules.configuration.properties.api.EncryptionAlgorithm;
import com.mulesoft.modules.configuration.properties.api.EncryptionMode;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Base64;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Provider to resolve secure properties.
 * <p>
 * This provider resolves properties that start with the prefix secure::, returning the value of the expression to the one
 * corresponding to the configuration file. In case the value is surrounded by the characters ![value], it will return the
 * decrypted value using the algorithm, mode and key provided.
 * </p>
 *
 * @since 1.0
 */
public class SecureConfigurationPropertiesProvider extends DefaultConfigurationPropertiesProvider {

  private final static String SECURE_PREFIX = "secure::";
  private final static Pattern SECURE_PATTERN = Pattern.compile("\\$\\{" + SECURE_PREFIX + "[^}]*}");

  private final EncryptionAlgorithm algorithm;
  private final EncryptionMode mode;
  private final boolean fileLevelEncryption;
  private final SecurePropertyPlaceholderModule securePropertyPlaceholderModule = new SecurePropertyPlaceholderModule();


  public SecureConfigurationPropertiesProvider(ResourceProvider resourceProvider, String file, EncryptionAlgorithm algorithm,
                                               String key, EncryptionMode mode, String encoding, boolean fileLevelEncryption) {
    super(file, encoding, resourceProvider);

    this.algorithm = algorithm;
    this.mode = mode;
    this.fileLevelEncryption = fileLevelEncryption;

    this.securePropertyPlaceholderModule.setEncryptionMode(mode);
    this.securePropertyPlaceholderModule.setEncryptionAlgorithm(algorithm);
    this.securePropertyPlaceholderModule.setKey(key);
  }

  @Override
  public Optional<ConfigurationProperty> getConfigurationProperty(String configurationAttributeKey) {
    if (configurationAttributeKey.startsWith(SECURE_PREFIX)) {
      String effectiveKey = configurationAttributeKey.substring(SECURE_PREFIX.length());

      ConfigurationProperty originalConfigurationProperty = configurationAttributes.get(effectiveKey);
      if (originalConfigurationProperty == null) {
        return empty();
      }
      String originalString = ((String) originalConfigurationProperty.getRawValue());
      String encryptedValue = originalString.substring(originalConfigurationProperty.getKey().length() + 1,
                                                       originalString.length() - algorithm.name().length() - mode.name().length()
                                                           - 2);
      final String decryptedValue = resolveInnerProperties(securePropertyPlaceholderModule.convertPropertyValue(encryptedValue));
      return of(new ConfigurationProperty() {

        @Override
        public Object getSource() {
          return originalConfigurationProperty.getSource();
        }

        @Override
        public Object getRawValue() {
          return decryptedValue;
        }

        @Override
        public String getKey() {
          return originalConfigurationProperty.getKey();
        }
      });
    } else {
      return empty();
    }
  }

  @Override
  public String getDescription() {
    ComponentLocation location = (ComponentLocation) getAnnotation(LOCATION_KEY);
    return format("<secure-properties file=\"%s\"> - file: %s, line number: %s", fileLocation,
                  location.getFileName().orElse(UNKNOWN),
                  location.getLineInFile().map(String::valueOf).orElse("unknown"));
  }

  @Override
  protected String createValue(String key, String value) {
    return format("%s:%s:%s:%s", key, value, algorithm, mode);
  }

  @Override
  protected InputStream getResourceInputStream(String file) throws IOException {
    InputStream originalStream = super.getResourceInputStream(file);
    if (!fileLevelEncryption) {
      return originalStream;
    }

    // We need to read the file, decrypt it, and create the input stream with its decrypted content
    byte[] content = IOUtils.toByteArray(originalStream);
    try {
      return new ByteArrayInputStream(securePropertyPlaceholderModule.decrypt(Base64.getDecoder().decode(content)));
    } catch (MuleEncryptionException e) {
      throw new IOException(e);
    }
  }

  private String resolveInnerProperties(String decryptedValue) {
    Matcher m = SECURE_PATTERN.matcher(decryptedValue);
    while (m.find()) {
      String secureInnerProperty = m.group();

      Optional<ConfigurationProperty> innerProperty = this.getConfigurationProperty(secureInnerProperty);
      if (innerProperty.isPresent()) {
        String innerPropertyValue = ((String) innerProperty.get().getRawValue());
        decryptedValue = decryptedValue.replaceAll(secureInnerProperty, innerPropertyValue);
      }
    }
    return decryptedValue;
  }
}
