/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.wss.internal.inbound;

import com.mulesoft.modules.wss.api.constants.SoapVersion;
import com.mulesoft.modules.wss.internal.auth.MustUnderstandAuthentication;
import com.mulesoft.modules.wss.internal.error.WssErrorTypeProvider;
import com.mulesoft.modules.wss.internal.error.WssException;
import com.mulesoft.modules.wss.internal.error.WssSecurityException;
import com.mulesoft.modules.wss.internal.util.SoapMessageHandler;
import org.apache.wss4j.common.ext.WSSecurityException;
import org.apache.wss4j.dom.WSConstants;
import org.apache.wss4j.dom.engine.WSSecurityEngineResult;
import org.apache.wss4j.dom.handler.WSHandlerResult;
import org.apache.wss4j.dom.message.WSSecHeader;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.security.Authentication;
import org.mule.runtime.api.security.SecurityException;
import org.mule.runtime.api.security.SecurityProviderNotFoundException;
import org.mule.runtime.api.security.UnknownAuthenticationTypeException;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.api.store.ObjectStoreException;
import org.mule.runtime.api.store.ObjectStoreManager;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Content;
import org.mule.runtime.extension.api.annotation.param.MediaType;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.parameter.CorrelationInfo;
import org.mule.runtime.extension.api.security.AuthenticationHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import javax.inject.Inject;
import javax.inject.Named;
import javax.xml.soap.SOAPMessage;
import java.io.InputStream;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.apache.wss4j.dom.util.WSSecurityUtil.findWsseSecurityHeaderBlock;
import static org.mule.runtime.api.metadata.MediaType.parse;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_STORE_MANAGER;

public class WssInboundOperations {

  private static final Logger LOGGER = LoggerFactory.getLogger(WssInboundOperations.class);
  private static final SoapMessageHandler ENVELOPE_DOCUMENT_FACTORY = new SoapMessageHandler();

  @Inject
  @Named(OBJECT_STORE_MANAGER)
  private ObjectStoreManager runtimeObjectStoreManager;

  @Throws(WssErrorTypeProvider.class)
  @MediaType(value = "application/xml", strict = false)
  @DisplayName("Validate WSS")
  public Result<InputStream, Void> validateWss(@Config WssInboundConfig config,
                                               @Content TypedValue<InputStream> request,
                                               @Optional(defaultValue = "SOAP_12") SoapVersion version,
                                               CorrelationInfo correlationInfo,
                                               AuthenticationHandler handler) {
    try {
      setMustUnderstand(handler);
      String contentType = request.getDataType().getMediaType().toRfcString();
      SOAPMessage soapMessage = ENVELOPE_DOCUMENT_FACTORY.create(request.getValue(), contentType, version);

      Document envelope = soapMessage.getSOAPPart().getDocumentElement().getOwnerDocument();
      Element header = findWsseSecurityHeaderBlock(envelope, envelope.getDocumentElement(), config.getActor(), false);

      WSHandlerResult result;

      if (header == null) {
        throw new WssSecurityException("Missing wsse:Security header in request");
      }
      result = config.processSecurity(envelope);
      saveSigningCert(correlationInfo.getCorrelationId(), result);
      if (config.extractSecurityHeader()) {
        new WSSecHeader(config.getActor(), envelope).removeSecurityHeader();
      }

      return Result.<InputStream, Void>builder()
          .output(ENVELOPE_DOCUMENT_FACTORY.serialize(soapMessage))
          .mediaType(parse(contentType))
          .build();
    } catch (WSSecurityException | WssException e) {
      throw new WssSecurityException("Error trying to process message security", e);
    }
  }

  private void saveSigningCert(String correlationId, WSHandlerResult result) {
    List<WSSecurityEngineResult> wsSecEngineResults = result.getResults();
    ObjectStore<Certificate> defaultPartition = runtimeObjectStoreManager.getDefaultPartition();
    for (WSSecurityEngineResult wser : wsSecEngineResults) {
      Integer wserAction = (Integer) wser.get(WSSecurityEngineResult.TAG_ACTION);
      if (wserAction != null && wserAction.intValue() == WSConstants.SIGN) {
        Certificate cert = (X509Certificate) wser.get(WSSecurityEngineResult.TAG_X509_CERTIFICATE);
        try {
          defaultPartition.store(correlationId, cert);
        } catch (ObjectStoreException e) {
          LOGGER.warn("Unable to store signing certificate for future encryption");
        }
        return;
      }
    }
  }

  private void setMustUnderstand(AuthenticationHandler handler) {
    if (handler == null) {
      return;
    }
    java.util.Optional<Authentication> authenticationOption = handler.getAuthentication();
    if (authenticationOption.isPresent()) {
      Authentication authentication = authenticationOption.get();
      // making this mutable since other impls can abuse of the getProperties method.
      Map<String, Object> newProps = new HashMap<>(authentication.getProperties());
      newProps.putAll(new MustUnderstandAuthentication().getProperties());
      authentication.setProperties(newProps);
    } else {
      try {
        handler.setAuthentication(new MustUnderstandAuthentication());
      } catch (SecurityProviderNotFoundException | SecurityException | UnknownAuthenticationTypeException e) {
        throw new WssException("Cannot set Must Understand Authentication", e);
      }
    }
  }
}
