/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.wss.internal.util;

import com.google.common.collect.ImmutableMap;
import com.mulesoft.modules.wss.api.constants.SoapVersion;
import com.mulesoft.modules.wss.internal.error.WssException;

import javax.xml.soap.MessageFactory;
import javax.xml.soap.MimeHeaders;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPMessage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import static com.mulesoft.modules.wss.api.constants.SoapVersion.SOAP_11;
import static com.mulesoft.modules.wss.api.constants.SoapVersion.SOAP_12;
import static javax.xml.soap.SOAPConstants.SOAP_1_1_PROTOCOL;
import static javax.xml.soap.SOAPConstants.SOAP_1_2_PROTOCOL;
import static org.apache.wss4j.common.util.AttachmentUtils.MIME_HEADER_CONTENT_TYPE;

public final class SoapMessageHandler {

  private static final Map<SoapVersion, MessageFactory> factories = ImmutableMap.of(SOAP_11, factoryFor(SOAP_1_1_PROTOCOL),
                                                                                    SOAP_12, factoryFor(SOAP_1_2_PROTOCOL));
  private static final String invalidInputMessage = "Unable to obtain SOAP request from payload";

  private static MessageFactory factoryFor(String protocol) {
    try {
      return MessageFactory.newInstance(protocol);
    } catch (SOAPException e) {
      throw new WssException("Error creating SOAP MESSAGE Factory for protocol: " + protocol);
    }
  }

  public SOAPMessage create(InputStream request, String contentType, SoapVersion version) {
    try {

      MessageFactory factory = factories.get(version);
      if (factory == null) {
        throw new WssException("Invalid SOAP Version [" + version + "]");
      }
      MimeHeaders headers = new MimeHeaders();
      headers.addHeader(MIME_HEADER_CONTENT_TYPE, contentType);
      SOAPMessage soapMessage = factory.createMessage(headers, request);
      if (soapMessage.getSOAPPart().getDocumentElement() == null) {
        throw new WssException(invalidInputMessage);
      }
      return soapMessage;
    } catch (IOException | SOAPException e) {
      throw new WssException(invalidInputMessage, e);
    }
  }

  public InputStream serialize(SOAPMessage message) {
    try {
      ByteArrayOutputStream out = new ByteArrayOutputStream();
      message.writeTo(out);
      return new ByteArrayInputStream(out.toByteArray());
    } catch (Exception e) {
      throw new WssException("Error while serializing result SOAP MESSAGE");
    }
  }
}
