/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.wss.internal.handler;

import com.mulesoft.modules.wss.api.constants.SignatureAlgorithmConstants;
import com.mulesoft.modules.wss.api.constants.SignatureC14nAlgorithmConstants;
import com.mulesoft.modules.wss.api.constants.SignatureDigestAlgorithmConstants;
import com.mulesoft.modules.wss.api.constants.SignatureKeyIdentifierConstants;
import com.mulesoft.modules.wss.api.outbound.BaseOutboundConfig;
import com.mulesoft.modules.wss.api.outbound.EncryptionConfig;
import com.mulesoft.modules.wss.api.outbound.MessagePart;
import com.mulesoft.modules.wss.api.outbound.SignatureConfig;
import com.mulesoft.modules.wss.api.store.KeyStoreConfiguration;
import com.mulesoft.modules.wss.internal.error.WssApplyException;
import com.mulesoft.modules.wss.internal.error.WssException;
import com.mulesoft.modules.wss.internal.error.WssSecurityException;
import org.apache.wss4j.common.WSEncryptionPart;
import org.apache.wss4j.common.crypto.Crypto;
import org.apache.wss4j.common.crypto.Merlin;
import org.apache.wss4j.dom.message.WSSecEncrypt;
import org.apache.wss4j.dom.message.WSSecHeader;
import org.apache.wss4j.dom.message.WSSecSignature;
import org.apache.xml.security.Init;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.api.store.ObjectStoreException;
import org.mule.runtime.api.store.ObjectStoreManager;

import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;

import static com.mulesoft.modules.wss.internal.handler.StoreConfigHandler.getWssProperties;
import static java.util.stream.Collectors.toList;

/**
 * Sets up the WSS4J Security Header configuration based on the the outbound configuration.
 */
public class OutboundConfigHandler {

  private WSSecHeader securityHeader;

  public OutboundConfigHandler(WSSecHeader securityHeader) {
    this.securityHeader = securityHeader;
  }

  public void handle(EncryptionConfig encryptionConfig, ObjectStoreManager runtimeObjectStoreManager, String correlationId) {
    try {
      Init.init();
      WSSecEncrypt wsEncrypt = new WSSecEncrypt(securityHeader);
      KeyStoreConfiguration keyStoreConfiguration = encryptionConfig.getKeyStoreConfiguration();
      if (keyStoreConfiguration == null) {
        X509Certificate signReqCertificate =
            getSigningRequestCertificate(correlationId, runtimeObjectStoreManager);
        if (signReqCertificate != null) {
          wsEncrypt.setUseThisCert(signReqCertificate);
        }
      } else {
        wsEncrypt.setUserInfo(keyStoreConfiguration.getAlias(), keyStoreConfiguration.getKeyPassword());
      }
      if (encryptionConfig.getEncryptionKeyIdentifier() != null) {
        wsEncrypt.setKeyIdentifierType(encryptionConfig.getEncryptionKeyIdentifier().getNumVal());
      }
      wsEncrypt.setSymmetricEncAlgorithm(encryptionConfig.getEncryptionSymAlgorithm().toString());
      wsEncrypt.setKeyEncAlgo(encryptionConfig.getEncryptionKeyTransportAlgorithm().toString());
      wsEncrypt.setDigestAlgorithm(encryptionConfig.getEncryptionDigestAlgorithm().toString());
      List<WSEncryptionPart> wsParts = createWSParts(encryptionConfig);
      if (!wsParts.isEmpty()) {
        wsEncrypt.getParts().addAll(wsParts);
      }
      wsEncrypt.build(keyStoreConfiguration != null ? getCrypto(keyStoreConfiguration) : null);
    } catch (Exception e) {
      throw new WssApplyException(e);
    }
  }

  public void handle(SignatureConfig signatureConfig) {
    try {
      WSSecSignature wssSign = new WSSecSignature(securityHeader);
      KeyStoreConfiguration keyStoreConfiguration = signatureConfig.getKeyStoreConfiguration();
      wssSign.setUserInfo(keyStoreConfiguration.getAlias(), keyStoreConfiguration.getKeyPassword());
      SignatureKeyIdentifierConstants keyIdentifier = signatureConfig.getKeyIdentifier();
      if (keyIdentifier != null) {
        wssSign.setKeyIdentifierType(keyIdentifier.getNumVal());
      }
      SignatureAlgorithmConstants algorithm = signatureConfig.getAlgorithm();
      if (algorithm != null) {
        wssSign.setSignatureAlgorithm(algorithm.toString());
      }
      SignatureC14nAlgorithmConstants c14nAlgorithm = signatureConfig.getC14nAlgorithm();
      if (c14nAlgorithm != null) {
        wssSign.setSigCanonicalization(c14nAlgorithm.toString());
      }
      SignatureDigestAlgorithmConstants digestAlgorithm = signatureConfig.getDigestAlgorithm();
      if (digestAlgorithm != null) {
        wssSign.setDigestAlgo(digestAlgorithm.toString());
      }

      List<WSEncryptionPart> wsParts = createWSParts(signatureConfig);
      if (!wsParts.isEmpty()) {
        wssSign.getParts().addAll(wsParts);
      }

      wssSign.build(getCrypto(keyStoreConfiguration));
    } catch (Exception e) {
      throw new WssSecurityException("Could not sign Envelope: " + e.getMessage(), e);
    }
  }

  private X509Certificate getSigningRequestCertificate(String correlationId, ObjectStoreManager runtimeObjectStoreManager) {
    ObjectStore<X509Certificate> defaultPartition = runtimeObjectStoreManager.getDefaultPartition();
    try {
      return defaultPartition.retrieve(correlationId);
    } catch (ObjectStoreException e) {
      return null;
    }
  }

  private List<WSEncryptionPart> createWSParts(BaseOutboundConfig baseOutboundConfig) {
    List<MessagePart> wssParts = baseOutboundConfig.getWssParts();
    if (wssParts == null) {
      return new ArrayList<>();
    }
    return wssParts.stream()
        .map(p -> new WSEncryptionPart(p.getLocalname(), p.getNamespace(), p.getEncode().toString()))
        .collect(toList());
  }

  private Crypto getCrypto(KeyStoreConfiguration keyStoreConfiguration) {
    try {
      return new Merlin(getWssProperties(keyStoreConfiguration), getClass().getClassLoader(), null);
    } catch (Exception e) {
      throw new WssException("Could not create crypto for keystore: " + keyStoreConfiguration.getPath(), e);
    }
  }

}
