/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.wss.internal.handler;

import com.mulesoft.modules.wss.api.constants.SamlConfirmationMethod;
import com.mulesoft.modules.wss.api.inbound.AuthenticateUserConfig;
import com.mulesoft.modules.wss.api.inbound.CredentialsConfig;
import com.mulesoft.modules.wss.api.inbound.DecryptionConfig;
import com.mulesoft.modules.wss.api.inbound.LDAPConfig;
import com.mulesoft.modules.wss.api.inbound.VerifySamlConfig;
import com.mulesoft.modules.wss.api.inbound.VerifySignatureConfig;
import com.mulesoft.modules.wss.api.inbound.VerifyTimestampConfig;
import com.mulesoft.modules.wss.api.inbound.VerifyUsernameTokenConfig;
import com.mulesoft.modules.wss.api.store.KeyStoreConfiguration;
import com.mulesoft.modules.wss.internal.error.MissingCertificateException;
import com.mulesoft.modules.wss.internal.error.WssException;
import com.mulesoft.modules.wss.internal.inbound.LDAPValidator;
import com.mulesoft.modules.wss.internal.inbound.SamlValidator;
import org.apache.wss4j.common.crypto.Merlin;
import org.apache.wss4j.common.ext.WSSecurityException;
import org.apache.wss4j.dom.WSConstants;
import org.apache.wss4j.dom.engine.WSSConfig;
import org.apache.wss4j.dom.handler.RequestData;
import org.apache.wss4j.dom.processor.EncryptedKeyProcessor;
import org.apache.wss4j.dom.processor.SAMLTokenProcessor;
import org.apache.wss4j.dom.processor.SignatureProcessor;
import org.apache.wss4j.dom.processor.TimestampProcessor;
import org.apache.wss4j.dom.processor.UsernameTokenProcessor;
import org.apache.wss4j.dom.validate.UsernameTokenValidator;
import org.apache.wss4j.dom.validate.Validator;
import org.springframework.security.ldap.DefaultSpringSecurityContextSource;
import org.springframework.security.ldap.authentication.BindAuthenticator;
import org.springframework.security.ldap.authentication.LdapAuthenticationProvider;
import org.springframework.security.ldap.search.FilterBasedLdapUserSearch;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

import static com.mulesoft.modules.wss.internal.handler.StoreConfigHandler.getWssProperties;
import static java.util.Collections.singletonList;
import static java.util.concurrent.TimeUnit.SECONDS;
import static org.apache.wss4j.dom.WSConstants.ENCRYPTED_KEY;
import static org.apache.wss4j.dom.WSConstants.SAML2_TOKEN;
import static org.apache.wss4j.dom.WSConstants.SAML_TOKEN;
import static org.apache.wss4j.dom.WSConstants.SIGNATURE;
import static org.apache.wss4j.dom.WSConstants.USERNAME_TOKEN;

/**
 * Sets up the WSS4J Engine configuration based on the the inbound configuration.
 */
public class InboundConfigHandler {

  private final WSSConfig wssConfig;
  private final RequestData requestData;

  public InboundConfigHandler(WSSConfig wssConfig, RequestData requestData) {
    this.wssConfig = wssConfig;
    this.requestData = requestData;
  }

  public void handle(VerifySignatureConfig signatureConfig) {
    try {
      Merlin merlin =
          new Merlin(getWssProperties(signatureConfig.getTrustStoreConfig()), getClass().getClassLoader(), null);
      requestData.setSigVerCrypto(merlin);
      wssConfig.setProcessor(SIGNATURE, SignatureProcessor.class);
      if (signatureConfig.getSubjectPattern() != null) {
        requestData.setSubjectCertConstraints(singletonList(Pattern.compile(signatureConfig.getSubjectPattern())));
      }
      if (signatureConfig.getIssuerPattern() != null) {
        requestData.setIssuerDNPatterns(singletonList(Pattern.compile(signatureConfig.getIssuerPattern())));
      }
    } catch (WSSecurityException e) {
      throw new WssException("Error setting signature validation configuration: " + e.getMessage(), e);
    } catch (IOException e) {
      throw new MissingCertificateException("Unable to get certificate from TrustStore.", e);
    }
  }

  public void handle(DecryptionConfig decryptionConfig) {
    try {
      KeyStoreConfiguration keyStoreConfig = decryptionConfig.getKeyStoreConfig();
      CredentialsCallbackHandler cch = (CredentialsCallbackHandler) requestData.getCallbackHandler();
      cch.setDecryptionConfigCredentials(new CredentialsConfig(keyStoreConfig.getAlias(), keyStoreConfig.getKeyPassword()));
      requestData.setDecCrypto(new Merlin(getWssProperties(keyStoreConfig), getClass().getClassLoader(), null));
      wssConfig.setProcessor(ENCRYPTED_KEY, EncryptedKeyProcessor.class);
    } catch (WSSecurityException e) {
      throw new WssException("Error setting decrypt configuration: " + e.getMessage(), e);
    } catch (IOException e) {
      throw new MissingCertificateException("Unable to get certificate from Key Store.", e);
    }
  }

  public void handle(VerifyTimestampConfig timestampConfig) {
    requestData.setTimeStampTTL((int) timestampConfig.getTimeUnit().convert(timestampConfig.getTimeToLive(), SECONDS));
    requestData.setTimeStampStrict(timestampConfig.isStrict());
    requestData.setTimeStampFutureTTL((int) timestampConfig.getTimeUnit().convert(timestampConfig.getSkewTime(), SECONDS));
    requestData.setRequireTimestampExpires(timestampConfig.isRequireExpiresHeader());
    requestData.setPrecisionInMilliSeconds(timestampConfig.isPrecisionInMilliseconds());
    wssConfig.setProcessor(WSConstants.TIMESTAMP, TimestampProcessor.class);
  }

  public void handle(VerifySamlConfig samlConfig) {
    TimeUnit timeUnit = samlConfig.getTimeUnit();
    SamlValidator samlAssertionValidator = new SamlValidator(requestData);
    samlAssertionValidator.setTtl((int) timeUnit.convert(samlConfig.getTimeToLive(), SECONDS));
    samlAssertionValidator.setFutureTTL((int) timeUnit.convert(samlConfig.getSkewTime(), SECONDS));
    samlAssertionValidator.setValidateSignatureAgainstProfile(samlConfig.isValidateSignatureAgainstProfile());
    samlAssertionValidator.setRequireStandardSubjectConfirmationMethod(samlConfig.isRequireStandardSubjectConfirmationMethod());
    samlAssertionValidator.setRequireBearerSignature(samlConfig.isRequireBearerSignature());

    SamlConfirmationMethod method = samlConfig.getRequiredSubjectConfirmationMethod();
    if (method != null) {
      samlAssertionValidator
          .setRequiredSubjectConfirmationMethod(method.getMethodStringForSAML(samlConfig.getSamlVersion()));
    }
    wssConfig.setProcessor(SAML_TOKEN, SAMLTokenProcessor.class);
    wssConfig.setValidator(SAML_TOKEN, samlAssertionValidator);
    wssConfig.setProcessor(SAML2_TOKEN, SAMLTokenProcessor.class);
    wssConfig.setValidator(SAML2_TOKEN, samlAssertionValidator);
  }

  public void handle(VerifyUsernameTokenConfig usernameTokenConfig) {
    requestData.setAddUsernameTokenNonce(usernameTokenConfig.isCheckNonce());
    requestData.setAddUsernameTokenCreated(true);
    requestData.setUtTTL(usernameTokenConfig.getTimeToLive());
    wssConfig.setProcessor(USERNAME_TOKEN, UsernameTokenProcessor.class);
    AuthenticateUserConfig authenticateUserConfig = usernameTokenConfig.getAuthenticateUserConfig();
    if (authenticateUserConfig instanceof LDAPConfig) {
      wssConfig.setValidator(USERNAME_TOKEN, createLDAPValidator((LDAPConfig) authenticateUserConfig));
    } else if (authenticateUserConfig instanceof CredentialsConfig) {
      CredentialsCallbackHandler cch = (CredentialsCallbackHandler) requestData.getCallbackHandler();
      cch.setUsernameTokenCredentials((CredentialsConfig) authenticateUserConfig);
      wssConfig.setValidator(USERNAME_TOKEN, new UsernameTokenValidator());
    }
  }

  private Validator createLDAPValidator(LDAPConfig ldapConfig) {
    Map<String, Object> environmentProperties = new HashMap<>();

    environmentProperties.put("java.naming.ldap.version", "3");
    environmentProperties.put("com.sun.jndi.ldap.connect.pool", "true");
    environmentProperties.put("com.sun.jndi.ldap.connect.pool.maxsize", "10");
    environmentProperties.put("com.sun.jndi.ldap.connect.pool.prefsize", "5");
    environmentProperties.put("com.sun.jndi.ldap.connect.pool.initsize", "3");
    environmentProperties.put("com.sun.jndi.ldap.connect.timeout", "10000");
    environmentProperties.put("com.sun.jndi.ldap.connect.pool.timeout", "60000");
    environmentProperties.put("com.sun.jndi.ldap.connect.pool.protocol", "plain");

    DefaultSpringSecurityContextSource defaultSpringSecurityContextSource =
        new DefaultSpringSecurityContextSource(ldapConfig.getProviderUrl());
    defaultSpringSecurityContextSource.setUserDn(ldapConfig.getUserDn());
    defaultSpringSecurityContextSource.setPassword(ldapConfig.getPassword());
    defaultSpringSecurityContextSource.setBaseEnvironmentProperties(environmentProperties);

    defaultSpringSecurityContextSource.afterPropertiesSet();
    FilterBasedLdapUserSearch filterBasedLdapUserSearch =
        new FilterBasedLdapUserSearch(ldapConfig.getSearchBase(), ldapConfig.getSearchFilter(),
                                      defaultSpringSecurityContextSource);
    filterBasedLdapUserSearch.setSearchSubtree(ldapConfig.isSearchInSubtree());

    BindAuthenticator bindAuthenticator = new BindAuthenticator(defaultSpringSecurityContextSource);
    bindAuthenticator.setUserSearch(filterBasedLdapUserSearch);

    return new LDAPValidator(new LdapAuthenticationProvider(bindAuthenticator));
  }
}
