/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.wss.internal.inbound;

import com.mulesoft.modules.wss.api.inbound.DecryptionConfig;
import com.mulesoft.modules.wss.api.inbound.VerifySamlConfig;
import com.mulesoft.modules.wss.api.inbound.VerifySignatureConfig;
import com.mulesoft.modules.wss.api.inbound.VerifyTimestampConfig;
import com.mulesoft.modules.wss.api.inbound.VerifyUsernameTokenConfig;
import com.mulesoft.modules.wss.internal.error.WssException;
import com.mulesoft.modules.wss.internal.error.WssSecurityException;
import com.mulesoft.modules.wss.internal.handler.CredentialsCallbackHandler;
import com.mulesoft.modules.wss.internal.handler.InboundConfigHandler;
import com.mulesoft.modules.wss.internal.security.SoapWssSecurityProvider;
import org.apache.wss4j.common.ext.WSSecurityException;
import org.apache.wss4j.dom.engine.WSSConfig;
import org.apache.wss4j.dom.engine.WSSecurityEngine;
import org.apache.wss4j.dom.handler.RequestData;
import org.apache.wss4j.dom.handler.WSHandlerResult;
import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.core.api.security.SecurityManager;
import org.mule.runtime.core.api.security.SecurityProvider;
import org.mule.runtime.extension.api.annotation.Configuration;
import org.mule.runtime.extension.api.annotation.Expression;
import org.mule.runtime.extension.api.annotation.Operations;
import org.mule.runtime.extension.api.annotation.dsl.xml.ParameterDsl;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.Parameter;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Placement;
import org.w3c.dom.Document;

import javax.inject.Inject;
import javax.inject.Named;

import static org.apache.wss4j.dom.WSConstants.ENCRYPTED_KEY;
import static org.apache.wss4j.dom.WSConstants.SAML2_TOKEN;
import static org.apache.wss4j.dom.WSConstants.SAML_TOKEN;
import static org.apache.wss4j.dom.WSConstants.SIGNATURE;
import static org.apache.wss4j.dom.WSConstants.TIMESTAMP;
import static org.apache.wss4j.dom.WSConstants.USERNAME_TOKEN;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_SECURITY_MANAGER;

@Operations({WssInboundOperations.class})
@Configuration(name = "inbound")
public class WssInboundConfig implements Initialisable, Disposable {

  @Inject
  @Named(OBJECT_SECURITY_MANAGER)
  private SecurityManager securityManager;

  @Parameter
  @Optional
  private String actor;

  @Optional
  @Parameter
  @Expression(NOT_SUPPORTED)
  private boolean extractSecurityHeader;

  @Parameter
  @Optional
  @Expression(NOT_SUPPORTED)
  @ParameterDsl(allowReferences = false)
  @DisplayName("Validate Username Token")
  @Placement(tab = "Validation")
  private VerifyUsernameTokenConfig usernameConfig;

  @Parameter
  @Optional
  @Expression(NOT_SUPPORTED)
  @ParameterDsl(allowReferences = false)
  @DisplayName("Validate Signature")
  @Placement(tab = "Validation")
  private VerifySignatureConfig verifySignatureConfig;

  @Parameter
  @Optional
  @Expression(NOT_SUPPORTED)
  @ParameterDsl(allowReferences = false)
  @DisplayName("Decrypt Message")
  @Placement(tab = "Decryption")
  private DecryptionConfig decryptionConfig;

  @Parameter
  @Optional
  @Expression(NOT_SUPPORTED)
  @ParameterDsl(allowReferences = false)
  @DisplayName("Validate Timestamp")
  @Placement(tab = "Validation")
  private VerifyTimestampConfig timestampConfig;

  @Parameter
  @Optional
  @Expression(NOT_SUPPORTED)
  @ParameterDsl(allowReferences = false)
  @DisplayName("Validate SAML Assertion")
  @Placement(tab = "SAML")
  private VerifySamlConfig verifySamlConfig;

  private WSSecurityEngine engine;
  private RequestData requestData;

  @Override
  public void initialise() throws InitialisationException {
    validateConfig();
    doSetUpSecurityProvider();
    doSetUpEngine();
  }

  private void doSetUpSecurityProvider() throws InitialisationException {
    SecurityProvider provider = securityManager.getProvider(SoapWssSecurityProvider.ID);
    if (provider == null) {
      SoapWssSecurityProvider securityProvider = new SoapWssSecurityProvider();
      securityProvider.initialise();
      securityManager.addProvider(securityProvider);
    }
  }

  private void validateConfig() throws InitialisationException {
    if (verifySignatureConfig == null &&
        decryptionConfig == null &&
        timestampConfig == null &&
        usernameConfig == null &&
        verifySamlConfig == null) {
      throw new InitialisationException(createStaticMessage("No WSS config was found in config, at least one is required"), this);
    }
  }

  private void doSetUpEngine() throws InitialisationException {
    WSSecurityEngine engine = new WSSecurityEngine();
    WSSConfig wssConfig = engine.getWssConfig();
    initConfig(wssConfig);
    RequestData data = new RequestData();
    data.setCallbackHandler(new CredentialsCallbackHandler());

    try {
      handleInboundConfig(wssConfig, data);

      // Setting this to true in order to accept RSA15 as a valid algorithm even though it's deprecated, this could be
      // configurable for future versions.
      data.setAllowRSA15KeyTransportAlgorithm(true);

      data.setActor(actor);
      data.setWssConfig(wssConfig);
      requestData = data;
      this.engine = engine;
    } catch (WssException e) {
      throw new InitialisationException(e, this);
    }
  }

  /**
   * Builds up the configuration for the engine. The order in which the configurations are handled matters.
   *
   * @param wssConfig
   * @param data
   */
  private void handleInboundConfig(WSSConfig wssConfig, RequestData data) {
    InboundConfigHandler handler = new InboundConfigHandler(wssConfig, data);
    if (verifySignatureConfig != null) {
      handler.handle(verifySignatureConfig);
    }
    if (decryptionConfig != null) {
      handler.handle(decryptionConfig);
    }
    if (timestampConfig != null) {
      handler.handle(timestampConfig);
    }
    if (usernameConfig != null) {
      handler.handle(usernameConfig);
    }
    if (verifySamlConfig != null) {
      handler.handle(verifySamlConfig);
    }
  }

  private void initConfig(WSSConfig wssConfig) {
    wssConfig.setProcessor(SAML_TOKEN, NullProcessor.class);
    wssConfig.setProcessor(SAML2_TOKEN, NullProcessor.class);
    wssConfig.setProcessor(SIGNATURE, NullProcessor.class);
    wssConfig.setProcessor(TIMESTAMP, NullProcessor.class);
    wssConfig.setProcessor(USERNAME_TOKEN, NullProcessor.class);
    wssConfig.setProcessor(ENCRYPTED_KEY, NullProcessor.class);
  }

  public String getActor() {
    return actor;
  }

  public boolean extractSecurityHeader() {
    return extractSecurityHeader;
  }

  @Override
  public void dispose() {
    engine = null;
  }

  public synchronized WSHandlerResult processSecurity(Document envelope) {
    try {
      return engine.processSecurityHeader(envelope, requestData);
    } catch (WSSecurityException e) {
      throw new WssSecurityException("Error processing security: " + e.getMessage(), e);
    }
  }
}

