/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.wss.internal.outbound;

import com.mulesoft.modules.wss.api.constants.SoapVersion;
import com.mulesoft.modules.wss.api.outbound.EncryptionConfig;
import com.mulesoft.modules.wss.api.outbound.SignatureConfig;
import com.mulesoft.modules.wss.api.outbound.TimestampConfig;
import com.mulesoft.modules.wss.internal.error.WssApplyException;
import com.mulesoft.modules.wss.internal.error.WssErrorTypeProvider;
import com.mulesoft.modules.wss.internal.handler.OutboundConfigHandler;
import com.mulesoft.modules.wss.internal.util.SoapMessageHandler;
import org.apache.wss4j.dom.message.WSSecHeader;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.api.store.ObjectStoreException;
import org.mule.runtime.api.store.ObjectStoreManager;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Content;
import org.mule.runtime.extension.api.annotation.param.MediaType;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.runtime.operation.Result;
import org.mule.runtime.extension.api.runtime.parameter.CorrelationInfo;
import org.w3c.dom.Document;

import javax.inject.Inject;
import javax.inject.Named;
import javax.xml.soap.SOAPMessage;
import java.io.InputStream;
import java.security.cert.X509Certificate;

import static org.mule.runtime.api.metadata.MediaType.parse;
import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_STORE_MANAGER;

public class WssOutboundOperations {

  private static final SoapMessageHandler ENVELOPE_DOCUMENT_FACTORY = new SoapMessageHandler();

  @Inject
  @Named(OBJECT_STORE_MANAGER)
  private ObjectStoreManager runtimeObjectStoreManager;

  @Throws(WssErrorTypeProvider.class)
  @MediaType(value = "application/xml", strict = false)
  @DisplayName("Apply WSS")
  public Result<InputStream, Void> applyWss(@Config WssOutboundConfig config,
                                            @Content TypedValue<InputStream> response,
                                            @Optional(defaultValue = "SOAP_12") SoapVersion version,
                                            CorrelationInfo correlationInfo) {
    try {
      String contentType = response.getDataType().getMediaType().toRfcString();
      SOAPMessage soapMessage = ENVELOPE_DOCUMENT_FACTORY.create(response.getValue(), contentType, version);
      Document envelope = soapMessage.getSOAPPart().getDocumentElement().getOwnerDocument();

      WSSecHeader securityHeader = new WSSecHeader(envelope);
      securityHeader.insertSecurityHeader();
      securityHeader.setMustUnderstand(config.getMustUnderstand());
      applyConfigs(config, correlationInfo.getCorrelationId(), securityHeader);

      InputStream result = ENVELOPE_DOCUMENT_FACTORY.serialize(soapMessage);
      return Result.<InputStream, Void>builder()
          .output(result)
          .mediaType(parse(contentType))
          .build();
    } catch (Exception e) {
      throw new WssApplyException(e);
    }
  }

  private void applyConfigs(WssOutboundConfig config, String correlationId, WSSecHeader securityHeader) {
    SignatureConfig signatureConfig = config.getSignatureConfig();
    EncryptionConfig encryptionConfig = config.getEncryptionConfig();
    TimestampConfig timestampConfig = config.getOutboundTimestampConfig();
    OutboundConfigHandler outboundConfigHandler = new OutboundConfigHandler(securityHeader);
    if (timestampConfig != null) {
      outboundConfigHandler.handle(timestampConfig);
    }
    if (signatureConfig != null) {
      outboundConfigHandler.handle(signatureConfig);
    }
    if (encryptionConfig != null) {
      outboundConfigHandler.handle(encryptionConfig, runtimeObjectStoreManager, correlationId);
    }
    removeSigningRequestCertificate(correlationId);
  }

  private void removeSigningRequestCertificate(String correlationId) {
    ObjectStore<X509Certificate> defaultPartition = runtimeObjectStoreManager.getDefaultPartition();
    try {
      if (defaultPartition.contains(correlationId)) {
        defaultPartition.remove(correlationId);
      }
    } catch (ObjectStoreException e) {
      throw new WssApplyException(e);
    }
  }
}
