/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.wss.internal.inbound;

import com.mulesoft.modules.wss.api.constants.SoapVersion;
import com.mulesoft.modules.wss.api.inbound.VerifySamlConfig;
import com.mulesoft.modules.wss.internal.error.WssSecurityException;
import com.mulesoft.modules.wss.internal.util.SoapMessageHandler;
import org.apache.wss4j.dom.handler.WSHandlerResult;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runner.RunWith;
import org.mockito.junit.MockitoJUnitRunner;
import org.mule.runtime.core.api.util.IOUtils;
import org.mule.runtime.core.internal.security.DefaultMuleSecurityManager;
import org.w3c.dom.Document;

import javax.xml.soap.SOAPMessage;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import static com.mulesoft.modules.wss.api.constants.SamlConfirmationMethod.BEARER;
import static com.mulesoft.modules.wss.api.constants.SamlConfirmationMethod.HOLDER_OF_KEY;
import static com.mulesoft.modules.wss.api.constants.SamlVersion.SAML20;
import static com.mulesoft.modules.wss.api.constants.SoapVersion.SOAP_11;
import static org.mule.runtime.core.api.util.ClassUtils.setFieldValue;
import static org.mule.runtime.core.api.util.IOUtils.getResourceAsStream;

@RunWith(MockitoJUnitRunner.class)
public class VerifySamlTestCase {

  private WssInboundConfig wssInboundConfig;

  private VerifySamlConfig verifySamlConfig;

  @Rule
  public ExpectedException expected = ExpectedException.none();


  @Before
  public void setUp() throws Exception {
    wssInboundConfig = new WssInboundConfig();
    verifySamlConfig = new VerifySamlConfig();
    setFieldValue(wssInboundConfig, "verifySamlConfig", verifySamlConfig, false);
    setFieldValue(wssInboundConfig, "securityManager", new DefaultMuleSecurityManager(), false);
    setFieldValue(verifySamlConfig, "samlVersion", SAML20, false);
  }

  @Test
  public void validateSaml() throws Exception {
    processSecurity("unsigned-saml-valid.xml", SOAP_11);
  }

  @Test
  public void validateExpiredSaml() throws Exception {
    expected.expect(WssSecurityException.class);
    expected.expectMessage("Error processing security: SAML token security failure");

    processSecurity("unsigned-saml-with-instant.xml", SOAP_11, DateTime.now().minusDays(1));
  }

  @Test
  public void validateNotExpiredSamlWithSkewTime() throws Exception {
    setFieldValue(verifySamlConfig, "skewTime", 18000, false);

    processSecurity("unsigned-saml-with-instant.xml", SOAP_11, DateTime.now().plusHours(3));
  }

  @Test
  public void validateSamlRequireBearerSignature() throws Exception {
    setFieldValue(verifySamlConfig, "requireBearerSignature", true, false);
    expected.expect(WssSecurityException.class);
    expected.expectMessage("Error processing security: SAML token security failure");

    processSecurity("unsigned-saml-valid.xml", SOAP_11);
  }

  @Test
  public void validateSignatureAgainstProfileError() throws Exception {
    setFieldValue(verifySamlConfig, "validateSignatureAgainstProfile", true, false);

    processSecurity("signed-saml-valid.xml", SOAP_11);
  }

  @Test
  public void validateSamlWrongMethod() throws Exception {
    setFieldValue(verifySamlConfig, "requiredSubjectConfirmationMethod", HOLDER_OF_KEY, false);
    expected.expect(WssSecurityException.class);
    expected.expectMessage("Error processing security: SAML token security failure");

    processSecurity("unsigned-saml-valid.xml", SOAP_11);
  }

  @Test
  public void validateSamlCorrectMethod() throws Exception {
    setFieldValue(verifySamlConfig, "requiredSubjectConfirmationMethod", BEARER, false);

    processSecurity("unsigned-saml-valid.xml", SOAP_11);
  }

  @Test
  public void validateSignedSaml() throws Exception {
    processSecurity("signed-saml-valid.xml", SOAP_11);
  }

  @Test
  public void validateInvalidSignedSaml() throws Exception {
    expected.expect(WssSecurityException.class);
    expected.expectMessage("SAML signature validation failed");

    processSecurity("signed-saml-invalid.xml", SOAP_11);
  }

  protected WSHandlerResult processSecurity(String fileName, SoapVersion soapVersion) throws Exception {
    return processSecurity(fileName, soapVersion, DateTime.now());
  }

  protected WSHandlerResult processSecurity(String fileName, SoapVersion soapVersion, DateTime instant) throws Exception {
    wssInboundConfig.initialise();

    String contentType = soapVersion.equals(SOAP_11) ? "text/xml" : "application/soap+xml";
    String request = getRequest(fileName);
    InputStream is = new ByteArrayInputStream(
                                              request
                                                  .replace("##Instant##", instant.withZone(DateTimeZone.getDefault()).toString())
                                                  .replace("##NotBefore##",
                                                           instant.withZone(DateTimeZone.getDefault()).toString())
                                                  .replace("##NotOnOrAfter##",
                                                           instant.plusHours(1).withZone(DateTimeZone.getDefault()).toString())
                                                  .getBytes());

    SOAPMessage soapMessage = new SoapMessageHandler().create(is, contentType, soapVersion);
    Document envelope = soapMessage.getSOAPPart().getDocumentElement().getOwnerDocument();

    return wssInboundConfig.processSecurity(envelope);
  }

  private String getRequest(String fileName) throws IOException {
    return IOUtils.toString(getResourceAsStream("munit/requests/" + fileName, VerifySamlTestCase.class));
  }

}
