/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.modules.oauth2.provider;

import static com.mulesoft.modules.oauth2.provider.api.Constants.HTTP_AUTHORIZATION_SCHEME_BEARER;
import static com.mulesoft.modules.oauth2.provider.api.Constants.RequestGrantType.PASSWORD;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.UNAUTHORIZED;
import static org.mule.runtime.http.api.HttpHeaders.Names.AUTHORIZATION;
import static org.mule.runtime.http.api.HttpHeaders.Names.WWW_AUTHENTICATE;

import java.io.IOException;
import java.util.Collections;
import java.util.Map;

import net.smartam.leeloo.client.request.OAuthClientRequest;
import net.smartam.leeloo.common.message.types.GrantType;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;

import org.junit.Test;

public class OAuth2ProviderModuleWithSpringResourceOwnerRolesTestCase extends AbstractOAuth2ProviderModuleTestCase
    implements SpringAwareOAuthProviderTest {

  @Override
  protected String doGetConfigFile() {
    return "oauth2-resource-owner-roles-http-config.xml";
  }

  @Override
  protected String getCommonConfigFile() {
    return "common-config-with-spring.xml";
  }

  @Override
  protected void doSetUp() throws Exception {
    super.doSetUp();
    client.getAuthorizedGrantTypes().add(PASSWORD);
    updateClientInOS();
  }

  @Test
  public void performResourceOwnerPasswordCredentialsGrantOAuth2DanceAndAccessProtectedResource()
      throws Exception {
    client.getAuthorizedGrantTypes().add(PASSWORD);

    final OAuthClientRequest oAuthClientRequest = OAuthClientRequest.tokenLocation(getTokenEndpointURL())
        .setGrantType(GrantType.PASSWORD)
        .setParameter("username", TEST_RESOURCE_OWNER_USERNAME)
        .setParameter("password", TEST_RESOURCE_OWNER_PASSWORD)
        .buildBodyMessage();

    oAuthClientRequest
        .setHeaders(Collections.singletonMap(AUTHORIZATION, getValidBasicAuthHeaderValue(TEST_CLIENT_ID, TEST_CLIENT_PASSWORD)));

    final PostMethod postToken = postOAuthClientRequestExpectingStatus(oAuthClientRequest, OK.getStatusCode());

    final Map<String, Object> tokenResponse = validateSuccessfulTokenResponseNoScopeNoRefresh(getContentAsMap(postToken));
    final String accessToken = (String) tokenResponse.get("access_token");

    accessRolelessProtectedResource(accessToken);

    accessRoledProtectedResource(accessToken);

    accessForbiddenRoledProtectedResource(accessToken);
  }

  private void accessRolelessProtectedResource(final String accessToken) throws Exception {
    final GetMethod getProtectedResource = new GetMethod(getProtectedResourceURL("/protected/no-role")
        + "?access_token=" + accessToken);
    executeHttpMethodExpectingStatus(getProtectedResource, OK.getStatusCode());
    assertThat(getProtectedResource.getResponseBodyAsString(), is(equalTo(PROTECTED_RESOURCE_CONTENT)));
  }

  private void accessRoledProtectedResource(final String accessToken) throws IOException {
    final GetMethod getProtectedResource = new GetMethod(getProtectedResourceURL("/protected/roles")
        + "?access_token=" + accessToken);
    executeHttpMethodExpectingStatus(getProtectedResource, OK.getStatusCode());
    assertThat(getProtectedResource.getResponseBodyAsString(), is(equalTo(PROTECTED_RESOURCE_CONTENT)));
  }

  private void accessForbiddenRoledProtectedResource(final String accessToken) throws IOException {
    final GetMethod getProtectedResource = new GetMethod(
                                                         getProtectedResourceURL("/protected/forbidden-role") + "?access_token="
                                                             + accessToken);
    executeHttpMethodExpectingStatus(getProtectedResource, UNAUTHORIZED.getStatusCode());
    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE), is(not(nullValue())));
    assertThat(getProtectedResource.getResponseHeader(WWW_AUTHENTICATE).getValue(),
               is(equalTo(HTTP_AUTHORIZATION_SCHEME_BEARER + " realm=\"OAuth2 Client Realm\"")));
  }
}
