/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.example.embedded.bootstrapper;

import static java.lang.ClassLoader.getSystemClassLoader;
import static java.lang.ModuleLayer.boot;
import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.lang.module.ModuleFinder.of;
import static java.lang.module.ModuleFinder.ofSystem;
import static java.nio.file.Files.walk;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

import static org.slf4j.LoggerFactory.getLogger;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.module.Configuration;
import java.lang.module.ModuleFinder;
import java.lang.module.ModuleReference;
import java.lang.reflect.Method;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.slf4j.Logger;

public class Main {

  private static final Logger LOGGER = getLogger(Main.class);

  private static final Path outputDirectory = Path.of(getProperty("user.dir"));

  public static void main(String[] args) throws Exception {
    LOGGER.info("Starting bootstrapper");

    // unzips the dependencies for the client
    unzipFile();

    final Set<String> bootModules = boot().modules().stream().map(Module::getName).collect(toSet());
    ModuleFinder finder = of(findLibsDirectory(outputDirectory));
    List<ModuleReference> rootModules = finder.findAll().stream()
        .filter(moduleRef -> !bootModules.contains(moduleRef.descriptor().name())).collect(toList());
    Path[] filteredModulesPaths = rootModules.stream().map(ModuleReference::location).filter(Optional::isPresent)
        .map(Optional::get).map(Paths::get).toArray(Path[]::new);
    ModuleFinder filteredModulesFinder = of(filteredModulesPaths);
    ModuleLayer parent = boot();
    final List<String> roots = rootModules.stream().map(moduleRef -> moduleRef.descriptor().name())
        .collect(toList());
    Configuration configuration = parent.configuration().resolve(filteredModulesFinder, ofSystem(), roots);

    ModuleLayer layer = parent.defineModulesWithOneLoader(configuration, getSystemClassLoader());
    openToModule(layer, "org.mule.runtime.jpms.utils", "java.base", asList("java.lang", "java.lang.reflect"));

    LOGGER.info("Embedded client's module layer built");

    Class<?> c = layer.findLoader("org.example.embedded.client").loadClass("org.example.embedded.Main");
    Method m = c.getDeclaredMethod("run");
    m.invoke(null);

    LOGGER.info("Embedded client execution finished");
  }

  private static Path findLibsDirectory(Path startPath) throws IOException {
    try (Stream<Path> paths = walk(startPath)) {
      return paths
          .filter(Files::isDirectory)
          .filter(path -> path.getFileName().toString().equals("lib"))
          .findFirst()
          .orElseThrow(() -> new RuntimeException(format("No 'lib' directory found under %s", startPath)));
    }
  }

  private static void unzipFile() {
    try {
      String fileZip = getProperty("embedded.client.dependencies.location");
      byte[] buffer = new byte[1024];
      ZipInputStream zis = new ZipInputStream(new FileInputStream(fileZip));
      ZipEntry zipEntry = zis.getNextEntry();
      while (zipEntry != null) {
        File newFile = newFile(outputDirectory.toFile(), zipEntry);
        if (zipEntry.isDirectory()) {
          if (!newFile.isDirectory() && !newFile.mkdirs()) {
            throw new IOException("Failed to create directory " + newFile);
          }
        } else {
          // fix for Windows-created archives
          File parent = newFile.getParentFile();
          if (!parent.isDirectory() && !parent.mkdirs()) {
            throw new IOException("Failed to create directory " + parent);
          }

          // write file content
          FileOutputStream fos = new FileOutputStream(newFile);
          int len;
          while ((len = zis.read(buffer)) > 0) {
            fos.write(buffer, 0, len);
          }
          fos.close();
        }
        zipEntry = zis.getNextEntry();
      }

      zis.closeEntry();
      zis.close();
    } catch (Exception e) {
      throw new RuntimeException(e.getMessage(), e);
    }
  }

  public static File newFile(File destinationDir, ZipEntry zipEntry) throws IOException {
    File destFile = new File(destinationDir, zipEntry.getName());

    String destDirPath = destinationDir.getCanonicalPath();
    String destFilePath = destFile.getCanonicalPath();

    if (!destFilePath.startsWith(destDirPath + File.separator)) {
      throw new IOException("Entry is outside of the target dir: " + zipEntry.getName());
    }

    return destFile;
  }

  public static void openToModule(ModuleLayer layer, String moduleName, String bootModuleName,
                                  List<String> packages) {
    layer.findModule(moduleName).ifPresent(module -> boot().findModule(bootModuleName).ifPresent(bootModule -> {
      for (String pkg : packages) {
        bootModule.addOpens(pkg, module);
      }
    }));
  }

}
