/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.example.embedded.utils;

import static org.mule.maven.client.api.MavenClientProvider.discoverProvider;
import static org.mule.maven.client.api.model.MavenConfiguration.newMavenConfigurationBuilder;
import static org.mule.maven.client.api.model.RemoteRepository.newRemoteRepositoryBuilder;

import static java.lang.System.getProperty;

import org.mule.maven.client.api.MavenClientProvider;
import org.mule.maven.client.api.SettingsSupplierFactory;
import org.mule.maven.client.api.model.MavenConfiguration.MavenConfigurationBuilder;

import java.io.File;
import java.io.IOException;
import java.net.URL;

public class MavenUtils {

  private static final String MULESOFT_PUBLIC_REPOSITORY = "https://repository.mulesoft.org/nexus/content/repositories/public/";
  private static final String LOCAL_REPO_URL = "maven.repo.local";
  private static final String DRY_RUN_REPO_ID = "mule.test.dryRunNexusId";
  private static final String DRY_RUN_REPO_URL = "mule.test.dryRunNexusUrl";
  private static final String MAVEN_CENTRAL = "https://repo.maven.apache.org/maven2/";

  /**
   * Creates a maven config builder with the default settings.
   *
   * @return a new maven configuration builder based on the user environment.
   */
  public static MavenConfigurationBuilder createDefaultMavenConfigurationBuilder()
      throws IOException {
    final MavenClientProvider mavenClientProvider =
        discoverProvider(MavenUtils.class.getClassLoader());
    File localMavenRepository;
    if (getProperty(LOCAL_REPO_URL) != null) {
      localMavenRepository = new File(getProperty(LOCAL_REPO_URL));
    } else {
      localMavenRepository =
        mavenClientProvider.getLocalRepositorySuppliers().environmentMavenRepositorySupplier().get();
    }

    MavenConfigurationBuilder mavenConfigurationBuilder =
        newMavenConfigurationBuilder()
            .forcePolicyUpdateNever(true)
            .forcePolicyUpdateAlways(false);

    mavenConfigurationBuilder.localMavenRepositoryLocation(localMavenRepository);

    final SettingsSupplierFactory settingsSupplierFactory = mavenClientProvider.getSettingsSupplierFactory();

    mavenConfigurationBuilder.remoteRepository(newRemoteRepositoryBuilder()
        .id("mulesoft-public")
        .url(new URL(MULESOFT_PUBLIC_REPOSITORY))
        .build());
    if (getProperty(DRY_RUN_REPO_URL) != null) {
      mavenConfigurationBuilder.remoteRepository(newRemoteRepositoryBuilder()
          .id(getProperty(DRY_RUN_REPO_ID, "mule-runtime-nexus-test-releases"))
          .url(new URL(getProperty(DRY_RUN_REPO_URL)))
          .build());
    }

    mavenConfigurationBuilder.remoteRepository(newRemoteRepositoryBuilder()
        .id("central")
        .url(new URL(MAVEN_CENTRAL))
        .build());

    settingsSupplierFactory.environmentUserSettingsSupplier().ifPresent(mavenConfigurationBuilder::userSettingsLocation);
    settingsSupplierFactory.environmentGlobalSettingsSupplier().ifPresent(mavenConfigurationBuilder::globalSettingsLocation);
    settingsSupplierFactory.environmentSettingsSecuritySupplier().ifPresent(mavenConfigurationBuilder::settingsSecurityLocation);

    return mavenConfigurationBuilder;
  }

}
