/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package com.mulesoft.mule.bootstrap;

import static java.lang.String.format;
import static java.util.Arrays.copyOfRange;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.joining;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;


/**
 * Just a very simple parser for command line arguments that has the features we need for this application, to avoid adding
 * dependencies with 3rd party libraries.
 * <p>
 * Options are required to be at the beginning, then all required positional arguments and then the additional optional arguments.
 * <p>
 * Currently, all options are optional.
 */
public class CommandLineParser {

  public static final String HELP_OPTION = "help";

  private final List<Option> validOptions = new ArrayList<>();
  private final List<String> requiredPositionalArguments = new ArrayList<>();

  private final Map<String, String> optionValues = new HashMap<>();
  private final List<String> requiredPositionalArgumentValues = new ArrayList<>();
  private String[] additionalArguments;

  public CommandLineParser() {
    validOptions.add(new Option(HELP_OPTION, "Show this help message and exit", false));
  }

  /**
   * Registers a valid option for the parser.
   *
   * @param name        The name of the option.
   * @param description A description for the help message.
   */
  public CommandLineParser withOption(String name, String description) {
    validOptions.add(new Option(name, description, true));
    return this;
  }

  /**
   * Registers a name for a required positional argument.
   *
   * @param name The name of the required positional argument.
   */
  public CommandLineParser withRequiredPositionalArgument(String name) {
    requiredPositionalArguments.add(name);
    return this;
  }

  /**
   * Parses the given arguments.
   *
   * @param args Arguments to parse.
   */
  public void parse(String... args) {
    // Resets in case the parser is being reused.
    optionValues.clear();
    requiredPositionalArgumentValues.clear();

    int firstNonOptionPos = 0;

    // Scans for options at the beginning.
    for (int i = 0; i < args.length; firstNonOptionPos = ++i) {
      if (!args[i].startsWith("-")) {
        // At the first non-option argument we'll stop looking for options.
        break;
      }

      parseOption(args[i]);
    }

    boolean validateRequiredArguments = !getBooleanOption(HELP_OPTION);

    // Scans for required positional arguments right after the options.
    for (int i = 0; i < requiredPositionalArguments.size(); i++) {
      if (firstNonOptionPos + i >= args.length) {
        if (validateRequiredArguments) {
          throw new IllegalArgumentException(format("Missing required argument %s", requiredPositionalArguments.get(i)));
        } else {
          break;
        }
      }
      requiredPositionalArgumentValues.add(args[firstNonOptionPos + i]);
    }

    // What remains are just additional arguments.
    additionalArguments = copyOfRange(args, firstNonOptionPos + requiredPositionalArgumentValues.size(), args.length);
  }

  /**
   * @param option The option name.
   * @return The value for the given option if it was present in the parsed arguments.
   */
  public Optional<String> getOptionValue(String option) {
    return ofNullable(optionValues.get(option));
  }

  /**
   * @param option The option name.
   * @return Whether the given option has been provided and the value represents a true according to
   *         {@link Boolean#parseBoolean(String)}.
   */
  public boolean getBooleanOption(String option) {
    return ofNullable(optionValues.get(option)).map(Boolean::parseBoolean).orElse(false);
  }

  /**
   * @param i The position of the desired positional argument.
   * @return The value for the positional argument at the given position.
   */
  public String getRequiredPositionalArgument(int i) {
    return requiredPositionalArgumentValues.get(i);
  }

  /**
   * @return All the additional arguments that were not identified as options or required positional arguments.
   */
  public String[] getAdditionalArguments() {
    return additionalArguments;
  }

  /**
   * Prints a helper text based on the registered options and required positional arguments.
   */
  public void printUsage() {
    String requiredPositionalArgumentsString = requiredPositionalArguments.stream()
        .map(s -> format("<%s> ", s))
        .collect(joining());

    System.out.println("Usage:");
    System.out.printf("%s [OPTIONS] %s[additionalArgs...]\n", WrapperlessArgumentsResolver.class.getSimpleName(),
                      requiredPositionalArgumentsString);
    System.out.println("Where options include:");
    for (Option option : validOptions) {
      if (option.isValueRequired()) {
        System.out.printf("\t--%s=<value>\t%s\n", option.getName(), option.getDescription());
      } else {
        System.out.printf("\t--%s\t%s\n", option.getName(), option.getDescription());
      }
    }
  }

  private void parseOption(String optionArg) {
    for (Option validOption : validOptions) {
      if (!validOption.matches(optionArg)) {
        continue;
      }

      String[] split = optionArg.split("=", 2);

      if (validOption.isValueRequired() && split.length != 2) {
        throw new IllegalArgumentException(format("Option %s requires a value", validOption.getName()));
      }
      optionValues.put(validOption.getName(), split.length != 2 ? "true" : split[1]);
      return;
    }

    throw new IllegalArgumentException(format("Invalid option: %s", optionArg));
  }

  private static class Option {

    private final String name;
    private final String description;
    private final boolean requiresValue;

    public Option(String name, String description, boolean requiresValue) {
      this.name = name;
      this.description = description;
      this.requiresValue = requiresValue;
    }

    public String getName() {
      return name;
    }

    public String getDescription() {
      return description;
    }

    public boolean isValueRequired() {
      return requiresValue;
    }

    public boolean matches(String optionArg) {
      return optionArg.equals("--" + name) || (requiresValue && optionArg.startsWith("--" + name + "="));
    }
  }
}
