/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package com.mulesoft.mule.bootstrap;

import static java.util.Collections.unmodifiableList;

import java.util.ArrayList;
import java.util.List;

/**
 * Provides access to the results of the arguments resolution for a wrapper-less execution.
 */
class WrapperlessArgumentsResolutionResult {

  private final List<String> options;
  private final String mainClass;
  private final List<String> mainClassArgs;

  WrapperlessArgumentsResolutionResult(List<String> options, String mainClass, List<String> mainClassArgs) {
    this.options = unmodifiableList(options);
    this.mainClass = mainClass;
    this.mainClassArgs = unmodifiableList(mainClassArgs);
  }

  /**
   * @return All the arguments concatenated in a unified list (in the proper order).
   */
  public List<String> getAllArguments() {
    List<String> allArguments = new ArrayList<>(options);

    // The main class needs to go after all the options
    allArguments.add(mainClass);

    // The arguments to the main class go after the main class name
    allArguments.addAll(mainClassArgs);

    return allArguments;
  }

  /**
   * @return Only the options to the JVM.
   */
  public List<String> getOptions() {
    return options;
  }

  /**
   * @return The main class argument and the arguments to the main class. Note that this can also have the shape of
   *         {@code --module=<module>[/<mainclass>]} as specified by the java command.
   */
  public List<String> getMainClassAndArgs() {
    List<String> mainClassAndArgs = new ArrayList<>(mainClassArgs.size() + 1);
    mainClassAndArgs.add(mainClass);
    mainClassAndArgs.addAll(mainClassArgs);
    return mainClassAndArgs;
  }
}
