/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package com.mulesoft.mule.bootstrap;

import static com.mulesoft.mule.bootstrap.CommandLineParser.HELP_OPTION;

import static java.io.File.pathSeparator;
import static java.lang.Boolean.parseBoolean;
import static java.lang.String.format;
import static java.lang.String.join;
import static java.lang.System.exit;
import static java.lang.System.getProperty;
import static java.nio.file.Files.find;
import static java.util.Arrays.asList;
import static java.util.Arrays.stream;
import static java.util.regex.Pattern.compile;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;

/**
 * Resolves the arguments that would be passed to the {@code java} process by an equivalent execution of
 * <a href="https://wrapper.tanukisoftware.com/doc/english/home.html">Tanuki's Java Service Wrapper</a>.
 * <p>
 * Using these arguments, it is possible to launch the Mule Container in a wrapper-less mode.
 * <p>
 * It can be used as a command line tool or as a library ({@link #resolve(String, String...)}).
 * <p>
 * Configurations are read from MULE_HOME/conf/wrapper.conf and MULE_HOME/conf/JAVA_VERSION/wrapper.jvmDependant.conf.
 * <p>
 * Additional arguments can be given which will be treated similarly as they would be by Tanuki's Wrapper.
 */
public class WrapperlessArgumentsResolver {

  private static final String JAVA_8_VERSION = "1.8";
  private static final String JAVA_RUNNING_VERSION = "java.specification.version";
  private static final String WRAPPER_ADDITIONAL_PREFIX = "wrapper.java.additional.";
  private static final String WRAPPER_JAVA_PREFIX = "wrapper.java.";
  private static final String WRAPPER_STRIP_QUOTES_SUFFIX = ".stripquotes";
  private static final Pattern ENV_VAR_REGEX = compile("%([^%]*)%");
  private static final String OPTIONS_OUT_FILE_OPTION = "OPTIONS_OUT_FILE";

  private final List<String> options = new ArrayList<>();
  private final List<String> classPaths = new ArrayList<>();
  private final List<String> libraryPaths = new ArrayList<>();
  private String mainClass = null;
  private final List<String> appArguments = new ArrayList<>();
  private final Function<String, String> envVarsResolver;

  public static void main(String[] args) {
    CommandLineParser commandLineParser = new CommandLineParser()
        .withOption(OPTIONS_OUT_FILE_OPTION, "A file to dump the resolved options to the java command")
        .withRequiredPositionalArgument("MULE_HOME");

    try {
      commandLineParser.parse(args);
    } catch (IllegalArgumentException e) {
      System.err.println(e.getMessage());
      commandLineParser.printUsage();
      exit(-1);
    }

    if (commandLineParser.getBooleanOption(HELP_OPTION)) {
      commandLineParser.printUsage();
      exit(0);
    }

    String muleHome = commandLineParser.getRequiredPositionalArgument(0);
    Optional<String> optionsOutFile = commandLineParser.getOptionValue(OPTIONS_OUT_FILE_OPTION);
    if (optionsOutFile.isPresent()) {
      // Dumps the options to a file
      WrapperlessArgumentsResolutionResult result = doResolve(null, muleHome, commandLineParser.getAdditionalArguments());
      try {
        dump(quote(result.getOptions()), optionsOutFile.get());
      } catch (FileNotFoundException e) {
        System.err.printf("Unable to dump options to file: %s", e.getMessage());
        exit(-2);
      }

      // ...And only the main class and its arguments to STDOUT
      System.out.println(join(" ", quote(result.getMainClassAndArgs())));
    } else {
      // Dumps everything to STDOUT
      System.out.println(join(" ", quote(resolve(muleHome, commandLineParser.getAdditionalArguments()))));
    }
  }

  /**
   * Resolves the arguments as explained in {@link WrapperlessArgumentsResolver}.
   *
   * @param muleHome       The path to the MULE_HOME. While a relative path can be used, it is recommended to use an absolute path
   *                       to avoid depending on the current working directory when launching the Mule Container.
   * @param additionalArgs Additional arguments which will be treated similarly as they would be by Tanuki's Wrapper.
   * @return The arguments that would be passed to the {@code java} process by an equivalent execution of
   *         <a href="https://wrapper.tanukisoftware.com/doc/english/home.html">Tanuki's Java Service Wrapper</a>.
   */
  public static List<String> resolve(String muleHome, String... additionalArgs) {
    return resolve(null, muleHome, additionalArgs);
  }

  /**
   * Allows to override the environment variables lookup, for testing purposes.
   *
   * @see #resolve(String, String...)
   */
  static List<String> resolve(Function<String, String> envVarsResolver, String muleHome, String... additionalArgs) {
    return doResolve(envVarsResolver, muleHome, additionalArgs).getAllArguments();
  }

  private static WrapperlessArgumentsResolutionResult doResolve(Function<String, String> envVarsResolver, String muleHome,
                                                                String... additionalArgs) {
    WrapperlessArgumentsResolver resolvedArgumentsBuilder = new WrapperlessArgumentsResolver(envVarsResolver);

    resolvedArgumentsBuilder.loadOptionsFromWrapperConf(muleHome);
    resolvedArgumentsBuilder
        .addOption("-Dmule.bootstrap.container.wrapper.class=org.mule.runtime.module.boot.internal.MuleContainerBasicWrapper");

    resolvedArgumentsBuilder.loadFromAdHocOptions(additionalArgs);

    // All additional arguments are passed as application arguments "as is"
    resolvedArgumentsBuilder.addAppArguments(asList(additionalArgs));

    resolvedArgumentsBuilder.loadOptionsFromJvmDependantWrapperConf(muleHome);

    return resolvedArgumentsBuilder.resolve();
  }

  private static void dump(List<String> args, String fileName) throws FileNotFoundException {
    try (PrintWriter writer = new PrintWriter(fileName)) {
      for (String arg : args) {
        writer.println(arg);
      }
    }
  }

  public WrapperlessArgumentsResolver(Function<String, String> envVarsResolver) {
    this.envVarsResolver = envVarsResolver != null ? envVarsResolver : System::getenv;
  }

  private void loadOptionsFromJvmDependantWrapperConf(String muleHome) {
    String jvmDependantPathPart;
    if (getProperty(JAVA_RUNNING_VERSION).startsWith(JAVA_8_VERSION)) {
      jvmDependantPathPart = "java8";
    } else {
      jvmDependantPathPart = "java11-plus";
    }
    Path wrapperJvmDepPath = Paths.get(muleHome, "conf", jvmDependantPathPart, "wrapper.jvmDependant.conf");
    loadFromConfFile(wrapperJvmDepPath);
  }

  private void loadOptionsFromWrapperConf(String muleHome) {
    Path wrapperConfPath = Paths.get(muleHome, "conf", "wrapper.conf");
    loadFromConfFile(wrapperConfPath);
  }

  private void addOption(String option) {
    options.add(option);
  }

  private void addClassPath(String classPath) {
    String expandedClassPath = expandClassPathWildcards(classPath);
    if (!expandedClassPath.isEmpty()) {
      classPaths.add(expandedClassPath);
    }
  }

  private void addLibraryPaths(String libraryPath) {
    libraryPaths.add(libraryPath);
  }

  private void setMainClass(String mainClass) {
    this.mainClass = mainClass;
  }

  private void addAppArguments(List<String> appArguments) {
    this.appArguments.addAll(appArguments);
  }

  public void loadFromConfFile(Path confFilePath) {
    Properties props = new Properties();
    try {
      props.load(new FileInputStream(confFilePath.toFile()));
    } catch (IOException e) {
      throw new RuntimeException(format("There was a problem resolving arguments from %s", confFilePath), e);
    }

    // TODO W-14150316: add module filtering to exclude the tanuki module

    for (Map.Entry<Object, Object> wrapperEntry : props.entrySet()) {
      String key = wrapperEntry.getKey().toString();
      String value = substituteEnvVariables(wrapperEntry.getValue().toString());
      if (key.startsWith(WRAPPER_ADDITIONAL_PREFIX) && !key.endsWith(WRAPPER_STRIP_QUOTES_SUFFIX)) {
        if (parseBoolean((props.getProperty(key + WRAPPER_STRIP_QUOTES_SUFFIX)))) {
          value = stripQuotes(value);
        }
        if (value.startsWith("--add-modules=")) {
          addOption(excludeTanukiModules(value));
        } else if (value.startsWith("--module-path=")) {
          addOption(excludeTanukiPaths(value));
        } else {
          addOption(value);
        }
      } else if (key.equals(WRAPPER_JAVA_PREFIX + "initmemory")) {
        addOption(format("-Xms%sm", value));
      } else if (key.equals(WRAPPER_JAVA_PREFIX + "maxmemory")) {
        addOption(format("-Xmx%sm", value));
      } else if (key.startsWith(WRAPPER_JAVA_PREFIX + "classpath.")) {
        addClassPath(value);
      } else if (key.startsWith(WRAPPER_JAVA_PREFIX + "library.path.")) {
        addLibraryPaths(value);
      } else if (key.equals(WRAPPER_JAVA_PREFIX + "mainclass")) {
        setMainClass(value);
      }
    }
  }

  public void loadFromAdHocOptions(String... options) {
    for (String adHocOption : options) {
      if (adHocOption.startsWith("-M") || adHocOption.startsWith("\"-M")) {
        // AdHoc options are always stripquoted, see AdditionalJvmParameters#writeAdHocProps
        // https://github.com/mulesoft/mule-distributions/blob/e2f1f67ed895ac805bd449600fb831974d394aa6/mule-wrapper-additional-parameters-parser/src/main/java/org/mule/runtime/params/AdditionalJvmParameters.java#L246
        addOption(stripQuotes(removeAdHocPrefix(adHocOption)));
      }
    }
  }

  public WrapperlessArgumentsResolutionResult resolve() {
    if (mainClass == null) {
      throw new IllegalArgumentException("Main Class missing");
    }

    List<String> allOptions = new ArrayList<>(options);

    // Adds the classpath paths joined by ':'
    if (!classPaths.isEmpty()) {
      allOptions.add("-classpath");
      allOptions.add(join(":", classPaths));
    }

    // Adds the library paths joined by ':'
    if (!libraryPaths.isEmpty()) {
      allOptions.add("-Djava.library.path=" + join(":", libraryPaths));
    }

    return new WrapperlessArgumentsResolutionResult(allOptions, mainClass, appArguments);
  }

  private String substituteEnvVariables(String propValue) {
    Matcher envVarMatcher = ENV_VAR_REGEX.matcher(propValue);

    // Cannot use StringBuilder because Matcher#appendReplacement with StringBuilder was added in Java 9
    StringBuffer sb = new StringBuffer();

    // Loops each match of the regex and attempts the replacement
    while (envVarMatcher.find()) {
      envVarMatcher.appendReplacement(sb, envVarLookUp(envVarMatcher.group(1)));
    }
    envVarMatcher.appendTail(sb);

    return sb.toString();
  }

  private String envVarLookUp(String envVarName) {
    String envVarValue = envVarsResolver.apply(envVarName);
    return envVarValue != null ? envVarValue : envVarName;
  }

  private static String removeAdHocPrefix(String arg) {
    if (arg.startsWith("\"-M")) {
      return arg.replaceFirst("\"-M", "\"");
    }
    return arg.substring(2);
  }

  private static String stripQuotes(String value) {
    // Trying to replicate this behaviour from Tanuki's Java Service Wrapper:
    // https://github.com/jonnyzzz/JavaServices/blob/f5c46ea7cb8e2dc860beff134e7a7bdd692949e1/wrapper/src/c/wrapper.c#L1327

    StringBuilder sb = new StringBuilder(value.length());
    int len = value.length();
    for (int i = 0; i < len; i++) {
      char curChar = value.charAt(i);
      if ((curChar == '\\') && (i < len - 1)) {
        char nextChar = value.charAt(i + 1);
        if (nextChar == '\\') {
          // Double backslash. Keep the first, and skip the second.
          sb.append(curChar);
          i++;
        } else if (nextChar == '\"') {
          // Escaped quote. Keep the quote.
          sb.append(nextChar);
          i++;
        } else {
          // Include the backslash as normal.
          sb.append(curChar);
        }
      } else if (curChar == '\"') {
        // Quote. Skip it.
      } else {
        sb.append(curChar);
      }
    }
    return sb.toString();
  }

  private String expandClassPathWildcards(String classPath) {
    // If there is no wildcard, returns the classPath as is.
    if (!classPath.contains("*")) {
      return classPath;
    }

    String[] classPathEntries = classPath.split(":");
    List<String> expandedClassPathEntries = new ArrayList<>(classPathEntries.length);
    for (String classPathEntry : classPathEntries) {
      expandedClassPathEntries.add(expandClassPathEntryWildcards(classPathEntry));
    }

    return join(":", expandedClassPathEntries);
  }

  private String expandClassPathEntryWildcards(String classPathEntry) {
    // If there is no wildcard, returns the classPathEntry as is.
    if (!classPathEntry.contains("*")) {
      return classPathEntry;
    }

    // Finds out the closest containing directory to use as root for the search
    FileSystem fs = FileSystems.getDefault();
    String pathBeforeWildcard = classPathEntry.split("\\*", 1)[0];
    Path basePath = Paths.get(pathBeforeWildcard);
    if (!pathBeforeWildcard.endsWith(fs.getSeparator())) {
      basePath = basePath.getParent();
    }

    // Creates a glob pattern to match the files against
    PathMatcher pathMatcher = fs.getPathMatcher(format("glob:%s", classPathEntry));

    // Performs the search
    try (Stream<Path> foundPaths = find(basePath, 1, (path, attrs) -> pathMatcher.matches(path))) {
      return foundPaths
          .map(Path::toString)
          .filter(not(this::isTanukiRelatedPath))
          .collect(joining(":"));
    } catch (IOException e) {
      throw new RuntimeException(format("Error expanding classpath entry: '%s'", classPathEntry), e);
    }
  }

  private boolean isTanukiRelatedPath(String path) {
    return path.contains("lib/boot/tanuki");
  }

  private boolean isTanukiRelatedModule(String moduleName) {
    return "org.mule.boot.tanuki".equals(moduleName);
  }

  private String excludeTanukiPaths(String modulePath) {
    return splitFilterAndJoin(modulePath.replace("WRAPPER_PATH_SEPARATOR", pathSeparator),
                              pathSeparator,
                              not(this::isTanukiRelatedPath));
  }

  private String excludeTanukiModules(String addModulesOption) {
    return splitFilterAndJoin(addModulesOption, ",", not(this::isTanukiRelatedModule));
  }

  private static List<String> quote(List<String> args) {
    return args.stream()
        .map(WrapperlessArgumentsResolver::quote)
        .collect(toList());
  }

  private static String quote(String arg) {
    return '"' + arg.replace("\"", "\\\"") + '"';
  }

  private static String splitFilterAndJoin(String originalValue, String delimiter, Predicate<String> filter) {
    String[] parts = originalValue.split("=", 2);
    return parts[0] + "=" + stream(parts[1].split(delimiter)).filter(filter).collect(joining(delimiter));
  }

  private static <T> Predicate<T> not(Predicate<T> original) {
    return original.negate();
  }
}
