/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.service.http.impl.service.client.ws;

import static com.mulesoft.service.http.impl.service.ws.WebSocketUtils.asVoid;
import static com.mulesoft.service.http.impl.service.ws.WebSocketUtils.streamInDataFrames;
import static java.lang.System.arraycopy;
import static java.util.Collections.unmodifiableList;
import static org.mule.runtime.api.metadata.MediaTypeUtils.isStringRepresentable;
import static org.mule.runtime.http.api.ws.WebSocket.WebSocketType.INBOUND;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.http.api.server.ws.WebSocketHandler;
import org.mule.runtime.http.api.server.ws.WebSocketRequest;
import org.mule.runtime.http.api.ws.WebSocket;
import org.mule.runtime.http.api.ws.WebSocketCloseCode;
import org.mule.runtime.http.api.ws.WebSocketProtocol;

import com.mulesoft.service.http.impl.service.ws.DataFrameEmitter;
import com.mulesoft.service.http.impl.service.ws.FragmentHandler;
import com.mulesoft.service.http.impl.service.ws.FragmentHandlerProvider;
import com.mulesoft.service.http.impl.service.ws.PipedFragmentHandlerProvider;

import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

import org.glassfish.grizzly.http.HttpRequestPacket;
import org.glassfish.grizzly.websockets.DefaultWebSocket;
import org.glassfish.grizzly.websockets.ProtocolHandler;
import org.glassfish.grizzly.websockets.WebSocketListener;

/**
 * Grizzly based implementation for an INBOUND {@link WebSocket}
 *
 * @since 1.3.0
 */
public class InboundWebSocket extends DefaultWebSocket implements WebSocket {

  private final String id;
  private final WebSocketHandler resource;
  private final WebSocketRequest request;
  private final Set<String> groups = new HashSet<>();
  private final FragmentHandlerProvider fragmentHandlerProvider;

  public InboundWebSocket(String id,
                          WebSocketHandler resource,
                          WebSocketRequest request,
                          ProtocolHandler protocolHandler,
                          HttpRequestPacket requestPacket,
                          WebSocketListener... listeners) {
    super(protocolHandler, requestPacket, listeners);
    this.id = id;
    this.resource = resource;
    this.request = request;
    fragmentHandlerProvider = new PipedFragmentHandlerProvider(id);
  }

  @Override
  public CompletableFuture<Void> send(InputStream content, MediaType mediaType) {
    DataFrameEmitter emitter = isStringRepresentable(mediaType)
        ? textEmitter()
        : binaryEmitter();

    return streamInDataFrames(content, emitter);
  }

  public FragmentHandler getFragmentHandler(Consumer<FragmentHandler> newFragmentHandlerCallback) {
    return fragmentHandlerProvider.getFragmentHandler(newFragmentHandlerCallback);
  }

  private DataFrameEmitter textEmitter() {
    return new DataFrameEmitter() {

      @Override
      public CompletableFuture<Void> stream(byte[] bytes, int offset, int len, boolean last) {
        return asVoid(InboundWebSocket.this.stream(last, new String(bytes, 0, len)));
      }

      @Override
      public CompletableFuture<Void> send(byte[] bytes, int offset, int len) {
        return asVoid(InboundWebSocket.this.send(new String(bytes, 0, len)));
      }
    };
  }

  private DataFrameEmitter binaryEmitter() {
    return new DataFrameEmitter() {

      @Override
      public CompletableFuture<Void> stream(byte[] bytes, int offset, int len, boolean last) {
        return asVoid(InboundWebSocket.this.stream(last, bytes, offset, len));
      }

      @Override
      public CompletableFuture<Void> send(byte[] bytes, int offset, int len) {
        if (offset != 0 || len != bytes.length) {
          byte[] aux = new byte[len];
          arraycopy(bytes, offset, aux, 0, len);
          bytes = aux;
        }
        return asVoid(InboundWebSocket.this.send(bytes));
      }
    };
  }

  @Override
  public List<String> getGroups() {
    synchronized (groups) {
      return unmodifiableList(new ArrayList<>(groups));
    }
  }

  @Override
  public void addGroup(String group) {
    synchronized (groups) {
      groups.add(group);
    }
  }

  @Override
  public void removeGroup(String group) {
    synchronized (groups) {
      groups.remove(group);
    }
  }

  @Override
  public CompletableFuture<Void> close(WebSocketCloseCode code, String reason) {
    return asVoid(completableClose(code.getProtocolCode(), reason));
  }

  public String getId() {
    return id;
  }

  @Override
  public URI getUri() {
    return request.getRequestUri();
  }

  @Override
  public WebSocketType getType() {
    return INBOUND;
  }

  @Override
  public WebSocketProtocol getProtocol() {
    return request.getScheme();
  }

  public WebSocketRequest getRequest() {
    return request;
  }

  public WebSocketHandler getResource() {
    return resource;
  }

  @Override
  public boolean equals(Object obj) {
    if (obj instanceof InboundWebSocket) {
      return id.equals(((InboundWebSocket) obj).getId());
    }

    return false;
  }

  @Override
  public int hashCode() {
    return id.hashCode();
  }

  @Override
  public String toString() {
    return "WebSocket Id: " + id + "\nType: " + INBOUND + "\nURI: " + getUri();
  }
}
