/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.service.http.impl.service.client.ws;

import static com.mulesoft.service.http.impl.service.ws.WebSocketUtils.asVoid;
import static java.lang.System.arraycopy;
import static java.util.Collections.unmodifiableList;
import static org.mule.runtime.api.metadata.MediaTypeUtils.isStringRepresentable;
import static org.mule.runtime.http.api.ws.WebSocket.WebSocketType.OUTBOUND;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.http.api.ws.WebSocket;
import org.mule.runtime.http.api.ws.WebSocketCloseCode;
import org.mule.runtime.http.api.ws.WebSocketProtocol;

import com.mulesoft.service.http.impl.service.ws.DataFrameEmitter;
import com.mulesoft.service.http.impl.service.ws.FragmentHandler;
import com.mulesoft.service.http.impl.service.ws.FragmentHandlerProvider;
import com.mulesoft.service.http.impl.service.ws.PipedFragmentHandlerProvider;
import com.mulesoft.service.http.impl.service.ws.WebSocketUtils;
import com.ning.http.client.providers.grizzly.websocket.GrizzlyWebSocketAdapter;

import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

/**
 * Grizzly based implementation of an OUTBOUND {@link WebSocket}
 *
 * @since 1.3.0
 */
public class OutboundWebSocket implements WebSocket {

  private final String id;
  private final URI uri;
  private final WebSocketProtocol protocol;
  private final Set<String> groups = new HashSet<>();
  private final GrizzlyWebSocketAdapter delegate;
  private final FragmentHandlerProvider fragmentHandlerProvider;

  public OutboundWebSocket(String id,
                           URI uri,
                           WebSocketProtocol protocol,
                           GrizzlyWebSocketAdapter delegate) {
    this.id = id;
    this.uri = uri;
    this.delegate = delegate;
    this.protocol = protocol;
    fragmentHandlerProvider = new PipedFragmentHandlerProvider(id);
  }

  @Override
  public CompletableFuture<Void> send(InputStream content, MediaType mediaType) {
    DataFrameEmitter emitter = isStringRepresentable(mediaType)
        ? textEmitter()
        : binaryEmitter();
    return WebSocketUtils.streamInDataFrames(content, emitter);
  }

  public FragmentHandler getFragmentHandler(Consumer<FragmentHandler> newFragmentHandlerCallback) {
    return fragmentHandlerProvider.getFragmentHandler(newFragmentHandlerCallback);
  }

  private DataFrameEmitter textEmitter() {
    return new DataFrameEmitter() {

      @Override
      public CompletableFuture<Void> stream(byte[] bytes, int offset, int len, boolean last) {
        return asVoid(delegate.completableStream(new String(bytes, offset, len), last));
      }

      @Override
      public CompletableFuture<Void> send(byte[] bytes, int offset, int len) {
        return asVoid(delegate.completableSend(new String(bytes, offset, len)));
      }
    };
  }

  private DataFrameEmitter binaryEmitter() {
    return new DataFrameEmitter() {

      @Override
      public CompletableFuture<Void> stream(byte[] bytes, int offset, int len, boolean last) {
        return asVoid(delegate.completableStream(bytes, offset, len, last));
      }

      @Override
      public CompletableFuture<Void> send(byte[] bytes, int offset, int len) {
        if (offset != 0 || len != bytes.length) {
          byte[] aux = new byte[len];
          arraycopy(bytes, offset, aux, 0, len);
          bytes = aux;
        }
        return asVoid(delegate.completableSend(bytes));
      }
    };
  }

  @Override
  public String getId() {
    return id;
  }

  @Override
  public List<String> getGroups() {
    synchronized (groups) {
      return unmodifiableList(new ArrayList<>(groups));
    }
  }

  @Override
  public void addGroup(String group) {
    synchronized (groups) {
      groups.add(group);
    }
  }

  @Override
  public void removeGroup(String group) {
    synchronized (groups) {
      groups.remove(group);
    }
  }

  @Override
  public CompletableFuture<Void> close(WebSocketCloseCode code, String reason) {
    return asVoid(delegate.close(code.getProtocolCode(), reason));
  }

  @Override
  public WebSocketType getType() {
    return OUTBOUND;
  }

  @Override
  public WebSocketProtocol getProtocol() {
    return protocol;
  }

  @Override
  public URI getUri() {
    return uri;
  }

  public org.glassfish.grizzly.websockets.WebSocket getGrizzlyWebSocket() {
    return delegate.getGrizzlyWebSocket();
  }

  @Override
  public boolean equals(Object obj) {
    if (obj instanceof OutboundWebSocket) {
      return id.equals(((OutboundWebSocket) obj).getId());
    }

    return false;
  }

  @Override
  public int hashCode() {
    return id.hashCode();
  }

  @Override
  public String toString() {
    return "WebSocket Id: " + id + "\nType: " + OUTBOUND + "\nURI: " + getUri();
  }
}
