/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.service.http.impl.service.client.ws;

import static java.lang.String.format;
import static org.glassfish.grizzly.websockets.WebSocket.NORMAL_CLOSURE;
import static org.mule.runtime.api.metadata.DataType.TEXT_STRING;
import static org.mule.runtime.api.metadata.MediaType.BINARY;
import static org.mule.runtime.http.api.ws.WebSocketCloseCode.fromProtocolCode;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.http.api.client.ws.WebSocketCallback;
import com.mulesoft.service.http.impl.service.ws.FragmentHandler;

import com.ning.http.client.HttpResponseBodyPart;
import com.ning.http.client.ws.DefaultWebSocketListener;
import com.ning.http.client.ws.WebSocket;
import com.ning.http.client.ws.WebSocketByteFragmentListener;
import com.ning.http.client.ws.WebSocketCloseCodeReasonListener;
import com.ning.http.client.ws.WebSocketTextFragmentListener;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * AHC adapter for a {@link WebSocketCallback}
 *
 * @since 1.3.0
 */
public class OutboundWebSocketListener extends DefaultWebSocketListener
    implements WebSocketByteFragmentListener, WebSocketTextFragmentListener, WebSocketCloseCodeReasonListener {

  private static final Logger LOGGER = LoggerFactory.getLogger(OutboundWebSocketListener.class);

  private final WebSocketCallback callback;
  private final String socketId;
  private volatile boolean connected = false;
  private OutboundWebSocket socket;

  public OutboundWebSocketListener(String socketId, WebSocketCallback callback) {
    this.socketId = socketId;
    this.callback = callback;
  }

  @Override
  public void onOpen(com.ning.http.client.ws.WebSocket webSocket) {
    super.onOpen(webSocket);
    synchronized (this) {
      connected = true;
      if (socket != null) {
        callback.onConnect(socket);
      }
    }
  }

  @Override
  public void onClose(WebSocket websocket, int code, String reason) {
    super.onClose(websocket);
    connected = false;
    callback.onClose(socket, fromProtocolCode(code), reason);
  }

  @Override
  public void onClose(com.ning.http.client.ws.WebSocket websocket) {
    onClose(webSocket, NORMAL_CLOSURE, "");
  }

  @Override
  public void onMessage(byte[] message) {
    callback.onMessage(socket, new TypedValue<>(new ByteArrayInputStream(message), DataType.builder()
        .type(InputStream.class)
        .mediaType(BINARY)
        .build()));
  }

  @Override
  public void onMessage(String message) {
    callback.onMessage(socket, new TypedValue<>(new ByteArrayInputStream(message.getBytes()), TEXT_STRING));
  }

  @Override
  public void onFragment(HttpResponseBodyPart fragment) {
    FragmentHandler handler = socket.getFragmentHandler(
                                                        h -> callback
                                                            .onMessage(socket,
                                                                       new TypedValue<>(h.getInputStream(), DataType.builder()
                                                                           .type(InputStream.class)
                                                                           .mediaType(BINARY)
                                                                           .build())));

    try {
      if (!handler.write(fragment.getBodyPartBytes())) {
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Incoming fragment for socket '{}' was discarded because the stream was already closed");
        }
      }
    } catch (IOException e) {
      if (LOGGER.isErrorEnabled()) {
        LOGGER.error(format(
                            "Error found while streaming data on socket '%s': %s. Stream will be closed", socket.getId(),
                            e.getMessage()),
                     e);
        handler.abort();
      }
      return;
    }

    if (fragment.isLast()) {
      handler.complete();
    }
  }

  @Override
  public void onError(Throwable t) {
    if (LOGGER.isErrorEnabled()) {
      LOGGER.error(String.format("The following error was catched for %sSocket '%s': %s",
                                 socket != null ? socket.getType().name() + " " : "",
                                 socketId,
                                 t.getMessage()),
                   t);
    }
  }

  public void setSocket(OutboundWebSocket socket) {
    synchronized (this) {
      this.socket = socket;
      if (connected) {
        callback.onConnect(socket);
      }
    }
  }
}
