/*
 * (c) 2003-2018 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.service.http.impl.service.ws;

import static java.lang.System.arraycopy;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static org.glassfish.grizzly.utils.Futures.completable;
import org.mule.runtime.api.util.concurrent.Latch;

import java.io.InputStream;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;

import org.glassfish.grizzly.GrizzlyFuture;

/**
 * Utility methods for implementing WebSockets
 *
 * @since 1.3.0
 */
public final class WebSocketUtils {

  public static final int DEFAULT_DATA_FRAME_SIZE = 8 * 1024;

  public static CompletableFuture<Void> streamInDataFrames(InputStream content, DataFrameEmitter emitter) {
    return streamInDataFrames(content, DEFAULT_DATA_FRAME_SIZE, emitter);
  }

  public static CompletableFuture<Void> streamInDataFrames(InputStream content, int frameSize, DataFrameEmitter emitter) {
    byte[] readBuffer = new byte[frameSize];
    byte[] writeBuffer = new byte[frameSize];
    int read;
    int write = 0;
    boolean streaming = false;
    CompletableFuture<Void> composedFuture = null;
    Latch latch = new Latch();
    AtomicReference<Throwable> error = new AtomicReference<>(null);

    try {
      while (error.get() == null && (read = content.read(readBuffer, 0, readBuffer.length)) != -1) {
        if (write > 0 && error.get() == null) {
          streaming = true;
          CompletableFuture frameFuture = emitter.stream(writeBuffer, 0, write, false).whenComplete((v, e) -> {
            if (e != null) {
              error.set(e);
              latch.release();
            }
          });
          if (composedFuture == null) {
            composedFuture = frameFuture;
          } else {
            composedFuture = composedFuture.thenCompose(v -> frameFuture);
          }
        }
        arraycopy(readBuffer, 0, writeBuffer, 0, read);
        write = read;
      }

      if (composedFuture != null) {
        composedFuture.whenComplete((v, e) -> latch.release());
        latch.await();
      }

      if (error.get() != null) {
        return failedFuture(error.get());
      }

      if (write == 0) {
        return completedFuture(null);
      }

      // because a bug in grizzly we need to create a byte array with the exact length
      if (write < writeBuffer.length) {
        byte[] exactSize = writeBuffer;
        writeBuffer = new byte[write];
        arraycopy(exactSize, 0, writeBuffer, 0, write);
      }

      if (error.get() != null) {
        return failedFuture(error.get());
      }

      if (streaming) {
        return emitter.stream(writeBuffer, 0, write, true);
      } else {
        return emitter.send(writeBuffer, 0, write);
      }
    } catch (Throwable t) {
      return failedFuture(t);
    }
  }

  public static <T> CompletableFuture<Void> asVoid(CompletableFuture<T> future) {
    CompletableFuture<Void> vf = new CompletableFuture<>();
    future.whenComplete((v, e) -> {
      if (e != null) {
        vf.completeExceptionally(e);
      } else {
        vf.complete(null);
      }
    });

    return vf;
  }

  public static <T> CompletableFuture<Void> asVoid(GrizzlyFuture<T> future) {
    return asVoid(completable(future));
  }

  public static <T> CompletableFuture<T> failedFuture(Throwable t) {
    CompletableFuture<T> future = new CompletableFuture<>();
    future.completeExceptionally(t);

    return future;
  }
}
