/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.service.http.impl.service.client.builder;

import static java.lang.System.getProperty;
import static java.util.Arrays.asList;

import org.mule.runtime.core.api.config.MuleProperties;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;

import com.mulesoft.service.http.impl.service.HostNameResolver;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.ning.http.client.RequestBuilder;
import com.ning.http.client.uri.Uri;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

/**
 * Specialization of {@link RequestBuilder} that handles the replacement of host names with a resolved ip address.
 */
public abstract class NameResolvingRequestBuilder extends RequestBuilder {

  private static final Logger logger = LoggerFactory.getLogger(NameResolvingRequestBuilder.class);

  private static final String DISABLE_ROUND_ROBIN = MuleProperties.SYSTEM_PROPERTY_PREFIX + "http.disableRoundRobin";

  private static List<String> disabledRoundRobinHosts = getProperty(DISABLE_ROUND_ROBIN) != null
      ? asList(getProperty(DISABLE_ROUND_ROBIN).split(",")) : Collections.<String>emptyList();

  private final HostNameResolver hostNameResolver;

  private final String host;

  private Iterator<InetAddress> resolvedAddressesIterator;

  public NameResolvingRequestBuilder(HttpRequest request, HostNameResolver hostNameResolver) {
    // url strings must already be properly encoded
    super(request.getMethod(), true);

    this.hostNameResolver = hostNameResolver;
    host = request.getUri().getHost();
  }

  protected void initAddressesIterator(IpListFactory factory) {
    List<InetAddress> addresses = null;
    try {
      addresses = factory.getAddresses();
    } catch (UnknownHostException e) {
      // In order to keep compatibility, let this exception be thrown later when the connection is
      // actually attempted by the AHC provider.
      logger.error("HostName '{}' could not be resolved ({})", host, e.getMessage());
    }
    resolvedAddressesIterator = addresses != null ? addresses.iterator() : Collections.<InetAddress>emptyIterator();
  }

  @Override
  public RequestBuilder setUrl(String url) {
    InetAddress nextResolvedAddresses = nextResolvedAddresses();
    if (nextResolvedAddresses != null) {
      setInetAddress(nextResolvedAddresses);
    }
    return super.setUrl(url);
  }

  @Override
  public RequestBuilder setUri(Uri uri) {
    InetAddress nextResolvedAddresses = nextResolvedAddresses();
    if (nextResolvedAddresses != null) {
      setInetAddress(nextResolvedAddresses);
    }
    return super.setUri(uri);
  }

  protected abstract List<InetAddress> getAddresses() throws UnknownHostException;

  protected abstract List<InetAddress> getRotateAddresses() throws UnknownHostException;

  public String getHost() {
    return host;
  }

  protected HostNameResolver getDomainNameResolver() {
    return hostNameResolver;
  }

  protected InetAddress nextResolvedAddresses() {
    if (resolvedAddressesIterator == null) {
      initAddressesIterator(disabledRoundRobinHosts.contains(host) ? new NonRotateIpListFactory() : new RotateIpListFactory());
    }
    return resolvedAddressesIterator.hasNext() ? resolvedAddressesIterator.next() : null;
  }

  /**
   * @return {@code true} if there are more available addresses left to try for this request.
   */
  public boolean hasNextResolvedAddresses() {
    if (resolvedAddressesIterator == null) {
      initAddressesIterator(new NonRotateIpListFactory());
    } else if (disabledRoundRobinHosts.contains(host)) {
      try {
        // Make it rotate, so next time it will not use the address that has recently failed
        getRotateAddresses();
      } catch (UnknownHostException e) {
        logger.debug("HostName '{}' could not be resolved ({})", host, e.getMessage());
      }
    }
    return resolvedAddressesIterator.hasNext();
  }

  public static void setDisabledRoundRobinHosts(List<String> hosts) {
    disabledRoundRobinHosts = hosts;
  }

  private interface IpListFactory {

    List<InetAddress> getAddresses() throws UnknownHostException;
  }

  private class RotateIpListFactory implements IpListFactory {

    @Override
    public List<InetAddress> getAddresses() throws UnknownHostException {
      return NameResolvingRequestBuilder.this.getRotateAddresses();
    }
  }

  private class NonRotateIpListFactory implements IpListFactory {

    @Override
    public List<InetAddress> getAddresses() throws UnknownHostException {
      return NameResolvingRequestBuilder.this.getAddresses();
    }
  }
}
