/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.service.http.impl.service.ws;

import static java.lang.String.format;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import org.mule.runtime.api.exception.MuleRuntimeException;

import java.io.IOException;
import java.io.InputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A {@link FragmentHandler} which is based on a {@link PipedInputStream}
 *
 * @since 1.3.0
 */
class PipedFragmentHandler implements FragmentHandler {

  private static final Logger LOGGER = LoggerFactory.getLogger(PipedFragmentHandler.class);

  private final String socketId;
  private final PipedOutputStream pipe;
  private final ManagedPipedInputStream stream;
  private final Runnable onClose;
  private boolean pipeClosed = false;

  /**
   * Creates a new instance
   *
   * @param socketId the id of the socket on which data is being sent
   * @param onClose  a callback to be invoked when {@code this} handler is closed or aborted.
   */
  public PipedFragmentHandler(String socketId, Runnable onClose) {
    try {
      stream = new ManagedPipedInputStream();
      pipe = new PipedOutputStream(stream);
      this.socketId = socketId;
      this.onClose = onClose;
    } catch (IOException e) {
      throw new MuleRuntimeException(createStaticMessage("Couldn't connect pipe to stream"), e);
    }
  }

  @Override
  public boolean write(byte[] data) throws IOException {
    if (!stream.isOpen()) {
      complete();
      return false;
    }

    if (pipeClosed) {
      return false;
    }

    pipe.write(data);
    return true;
  }

  @Override
  public void complete() {
    if (pipeClosed) {
      return;
    }

    try {
      pipe.close();
    } catch (IOException e) {
      LOGGER.error(format("Could not properly close streaming pipe for socket '%s'. %s", socketId, e.getMessage()), e);
    } finally {
      pipeClosed = true;
      onClose.run();
    }
  }

  @Override
  public void abort() {
    complete();

    try {
      stream.close();
    } catch (IOException e) {
      LOGGER.error(format("Could not properly close stream for socket '%s'. %s", socketId, e.getMessage()), e);
    } finally {
      onClose.run();
    }
  }

  @Override
  public InputStream getInputStream() {
    return stream;
  }
}
