/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.service.http.impl.service.client;

import static com.ning.http.client.AsyncHttpClientConfigDefaults.defaultUseProxyProperties;
import static org.mule.runtime.http.api.ws.WebSocketProtocol.forScheme;

import org.mule.runtime.api.scheduler.SchedulerConfig;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.client.proxy.ProxyConfig;
import org.mule.runtime.http.api.client.ws.WebSocketCallback;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.exception.InvalidStatusCodeException;
import org.mule.runtime.http.api.ws.WebSocket;
import org.mule.service.http.impl.service.client.GrizzlyHttpClient;

import com.mulesoft.service.http.impl.service.HostNameResolver;
import com.mulesoft.service.http.impl.service.client.builder.DirectNameResolvingRequestBuilder;
import com.mulesoft.service.http.impl.service.client.builder.NameResolvingRequestBuilder;
import com.mulesoft.service.http.impl.service.client.builder.NoNameResolvingRequestHandler;
import com.mulesoft.service.http.impl.service.client.builder.ProxyNameResolvingRequestBuilder;
import com.mulesoft.service.http.impl.service.client.ws.OutboundWebSocket;
import com.mulesoft.service.http.impl.service.client.ws.OutboundWebSocketListener;
import com.mulesoft.service.http.impl.service.client.ws.reconnect.OutboundWebSocketReconnectionHandler;

import java.io.IOException;
import java.net.ConnectException;
import java.net.SocketException;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import com.ning.http.client.AsyncHttpClient.BoundRequestBuilder;
import com.ning.http.client.AsyncHttpClientConfig.Builder;
import com.ning.http.client.ListenableFuture;
import com.ning.http.client.Request;
import com.ning.http.client.RequestBuilder;
import com.ning.http.client.providers.grizzly.websocket.GrizzlyWebSocketAdapter;
import com.ning.http.client.ws.BadStatusCodeException;
import com.ning.http.client.ws.WebSocketUpgradeHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Adds host name resolution and retry on network errors on each resolved address functionality to the base
 * {@link GrizzlyHttpClient}.
 * <p>
 * Even though the name {@code grizzly} is in the class name, it uses the AHC API and not grizzly stuff. It is just to mark it as
 * a specialization of Mule's {@link GrizzlyHttpClient}.
 * <p>
 * In order to optimize the retries (ensuring each retry is made against a different address each time), this object has state,
 * keeping for each request which addresses haven't been tried yet.
 */
public class EEGrizzlyHttpClient extends GrizzlyHttpClient {

  private static final Logger LOGGER = LoggerFactory.getLogger(EEGrizzlyHttpClient.class);
  private static boolean AHC_PROXY_PROPERTIES = defaultUseProxyProperties();

  private final HostNameResolver hostNameResolver;
  private final Map<HttpRequest, NameResolvingRequestBuilder> requestBuilders = new ConcurrentHashMap<>();

  /**
   * @param config           the http client configuration to use.
   * @param schedulerService the provider of the thread pools to be used by the Grizzly client
   * @param schedulersConfig the base configuration for the created inner schedulers.
   * @param hostNameResolver the object responsible on providing the addresses for each hostName.
   */
  public EEGrizzlyHttpClient(HttpClientConfiguration config,
                             SchedulerService schedulerService,
                             SchedulerConfig schedulersConfig,
                             HostNameResolver hostNameResolver) {
    super(config, schedulerService, schedulersConfig);
    this.hostNameResolver = hostNameResolver;
  }

  @Override
  protected void doConfigureProxy(Builder builder, ProxyConfig proxyConfig) {
    // Do not put the proxy config in the config for the client, it will be handled on a per request basis.
  }

  @Override
  protected RequestBuilder createRequestBuilder(HttpRequest request, HttpRequestOptions options,
                                                RequestConfigurer requestConfigurer)
      throws IOException {
    if (!requestBuilders.containsKey(request)) {
      final NameResolvingRequestBuilder requestBuilder = doCreateRequestBuilder(request, options);
      requestConfigurer.configure(requestBuilder);
      requestBuilders.put(request, requestBuilder);
    }

    return requestBuilders.get(request);
  }

  private NameResolvingRequestBuilder doCreateRequestBuilder(HttpRequest request, HttpRequestOptions options) {
    if (getProxyConfig() != null) {
      // Proxy config set up will be updated once the URI is set
      return new ProxyNameResolvingRequestBuilder(request, hostNameResolver, options.getProxyConfig().orElse(getProxyConfig()));
    } else if (AHC_PROXY_PROPERTIES) {
      return new NoNameResolvingRequestHandler(request, hostNameResolver);
    } else {
      return new DirectNameResolvingRequestBuilder(request, hostNameResolver);
    }
  }

  @Override
  public HttpResponse send(HttpRequest request, HttpRequestOptions options)
      throws IOException, TimeoutException {
    SocketException lastRetriableException;

    try {
      return super.send(request, options);
    } catch (SocketException e) {
      lastRetriableException = e;
      while (hasMoreAddressesToRetry(request)) {
        try {
          logAddressRetrying(request, e);
          return super.send(request, options);
        } catch (SocketException se) {
          // Ignore, the outer while loop will handle the retry
          lastRetriableException = se;
        }
      }
      logAddressesDepleted(request);
      throw lastRetriableException;
    } finally {
      requestBuilders.remove(request);
    }
  }

  @Override
  public CompletableFuture<HttpResponse> sendAsync(HttpRequest request, HttpRequestOptions options) {
    CompletableFuture<HttpResponse> future = new CompletableFuture<>();
    super.sendAsync(request, options).whenComplete((response, exception) -> {
      if (response != null) {
        requestBuilders.remove(request);
        future.complete(response);
      } else {
        if (exception instanceof SocketException) {
          if (hasMoreAddressesToRetry(request)) {
            logAddressRetrying(request, exception);
            sendAsync(request, options)
                .whenComplete((recResponse, recException) -> {
                  if (recResponse != null) {
                    future.complete(recResponse);
                  } else {
                    future.completeExceptionally(recException);
                  }
                });
          } else {
            logAddressesDepleted(request);
            requestBuilders.remove(request);
            future.completeExceptionally(exception);
          }
        } else {
          requestBuilders.remove(request);
          future.completeExceptionally(exception);
        }
      }
    });
    return future;
  }

  @Override
  public CompletableFuture<WebSocket> openWebSocket(HttpRequest request,
                                                    HttpRequestOptions requestOptions,
                                                    String socketId,
                                                    WebSocketCallback callback) {
    CompletableFuture<WebSocket> future = new CompletableFuture<>();
    try {
      final Request effectiveRequest = createGrizzlyRequest(request, requestOptions);

      final OutboundWebSocketListener socketListener = new OutboundWebSocketListener(socketId, callback);

      BoundRequestBuilder reqBuilder = asyncHttpClient.prepareRequest(effectiveRequest);
      WebSocketUpgradeHandler upgradeHandler = new WebSocketUpgradeHandler.Builder().addWebSocketListener(socketListener).build();
      ListenableFuture<com.ning.http.client.ws.WebSocket> listenable = reqBuilder.execute(upgradeHandler);
      listenable.addListener(() -> {
        try {
          com.ning.http.client.ws.WebSocket ws = listenable.get();

          OutboundWebSocketReconnectionHandler reconnectionHandler =
              new OutboundWebSocketReconnectionHandler(this, request, requestOptions, socketId, callback);

          OutboundWebSocket socket = new OutboundWebSocket(socketId,
                                                           request.getUri(),
                                                           forScheme(request.getUri().getScheme()),
                                                           (GrizzlyWebSocketAdapter) ws,
                                                           reconnectionHandler);

          socketListener.setSocket(socket);
          requestBuilders.remove(request);
          future.complete(socket);
        } catch (Throwable t) {
          if (t.getCause() instanceof ConnectException) {
            if (hasMoreAddressesToRetry(request)) {
              openWebSocket(request, requestOptions, socketId, callback)
                  .whenComplete((recResponse, recException) -> {
                    if (recResponse != null) {
                      future.complete(recResponse);
                    } else {
                      future.completeExceptionally(recException);
                    }
                  });
            } else {
              requestBuilders.remove(request);
              future.completeExceptionally(mapException(t));
            }
          } else {
            requestBuilders.remove(request);
            future.completeExceptionally(mapException(t));
          }
        }
      }, Runnable::run);
    } catch (Exception e) {
      future.completeExceptionally(mapException(e));
    }

    return future;
  }

  private Throwable mapException(Throwable t) {
    if (t instanceof ExecutionException) {
      t = t.getCause();
    }

    if (t instanceof BadStatusCodeException) {
      t = new InvalidStatusCodeException(((BadStatusCodeException) t).getStatus());
    }

    return t;
  }

  protected void logAddressRetrying(final HttpRequest request, Throwable e) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Retrying request to '{}' because of {} ({})", request.getUri(), e.getClass().getName(), e.getMessage());
    }
  }

  protected void logAddressesDepleted(final HttpRequest request) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Available addresses depleted for request to '{}'. Throwing Exception...", request.getUri());
    }
  }

  protected boolean hasMoreAddressesToRetry(final HttpRequest request) {
    return requestBuilders.get(request).hasNextResolvedAddresses();
  }

  public static void refreshSystemProperties() {
    AHC_PROXY_PROPERTIES = defaultUseProxyProperties();
  }
}
