/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.service.http.impl.service;

import static com.mulesoft.service.http.impl.AllureConstants.HttpFeature.HTTP_EE_SERVICE;
import static com.mulesoft.service.http.impl.AllureConstants.HttpFeature.HttpStory.DNS_RESOLUTION;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.Matchers.greaterThan;
import static org.hamcrest.Matchers.lessThanOrEqualTo;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.core.Every.everyItem;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;
import static org.reflections.ReflectionUtils.getFields;
import static org.reflections.ReflectionUtils.withName;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.tck.junit4.AbstractMuleTestCase;

import org.junit.BeforeClass;
import org.junit.Test;

import java.lang.reflect.Field;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(HTTP_EE_SERVICE)
@Story(DNS_RESOLUTION)
public class DomainNameResolverTestCase extends AbstractMuleTestCase {

  private static final String TEST_HOST = System.getProperty("domainNameResolver.test.host", "salesforce.com");
  private static final String UNKNOWN_TEST_HOST =
      System.getProperty("domainNameResolver.test.unknownHost", "unknown.mulesoft.unknown");

  /**
   * Validates that the host used for testing has at least 2 different IP addresses.
   * <p>
   * This can be validated in unix-like systems by running {@code 'dig <host> a'} and checking the returned ip addresses.
   *
   * @throws UnknownHostException see {@link #validateUnknownTestHost()}.
   */
  @BeforeClass
  public static void validateTestHost() throws UnknownHostException {
    assertThat(TEST_HOST + " doesn't resolve to at least 2 different ips and is no good for this test.",
               InetAddress.getAllByName(TEST_HOST).length, greaterThan(1));
  }

  /**
   * Validates that the unknown host used for testing is really unknown.
   * <p>
   * This can be validated by pinging that host and checking that the given address cannot be resolved.
   */
  @BeforeClass
  public static void validateUnknownTestHost() {
    try {
      InetAddress.getAllByName(UNKNOWN_TEST_HOST);
      fail(UNKNOWN_TEST_HOST + " could be resolved when it has to be unresolvable/unknown.");
    } catch (UnknownHostException e) {

    }
  }

  @Test
  public void allIpsResolved() throws UnknownHostException {
    HostNameResolver resolver = new HostNameResolver();

    assertThat(resolver.getAddresses(TEST_HOST), hasSize(InetAddress.getAllByName(TEST_HOST).length));
  }

  @Test(expected = UnknownHostException.class)
  public void unknownHost() throws UnknownHostException {
    HostNameResolver resolver = new HostNameResolver();
    resolver.getAddresses(UNKNOWN_TEST_HOST);
  }

  @Test
  public void cycleAllIps() throws UnknownHostException {
    HostNameResolver resolver = new HostNameResolver();

    for (int i = 0; i < resolver.getAddresses(TEST_HOST).size(); ++i) {
      assertThat(resolver.getAddresses(TEST_HOST), everyItem(not(nullValue(InetAddress.class))));
    }
  }

  @Test
  public void ringCounterOverflow() throws UnknownHostException {
    HostNameResolver resolver = new HostNameResolver(hostName -> {
      HostNameIpsRing hostNameIpsRing = new HostNameIpsRing(hostName);
      Field next = getFields(HostNameIpsRing.class, withName("idxAddresses")).iterator().next();
      next.setAccessible(true);
      try {
        ((AtomicInteger) next.get(hostNameIpsRing)).set(Integer.MAX_VALUE);
      } catch (IllegalArgumentException | IllegalAccessException e) {
        throw new MuleRuntimeException(e);
      }
      return hostNameIpsRing;
    });

    final List<InetAddress> resolvedAddresses = resolver.getAddresses(TEST_HOST);
    assertThat(resolvedAddresses.toString(), resolvedAddresses,
               hasSize(lessThanOrEqualTo(InetAddress.getAllByName(TEST_HOST).length)));
  }

  @Test
  public void nonRotatedIpAddresses() throws UnknownHostException {
    HostNameResolver resolver = new HostNameResolver();
    List<InetAddress> l1 = resolver.getAddresses(TEST_HOST);
    List<InetAddress> l2 = resolver.getAddresses(TEST_HOST);
    assertThat(l2, is(l1));
  }

  @Test
  public void rotatedAddress() throws UnknownHostException {
    HostNameResolver resolver = new HostNameResolver();
    List<InetAddress> l1 = resolver.getRotatedAddresses(TEST_HOST);
    List<InetAddress> l2 = resolver.getRotatedAddresses(TEST_HOST);
    assertThat(l2, hasSize(l2.size()));
    assertThat(l2, not(l1));

    // the addresses obtained are just a rotation of 1 element of the first one
    ArrayList<InetAddress> l1mutable = new ArrayList<>(l1);
    l1mutable.add(l1mutable.remove(0));
    assertThat(l1mutable, is(l2));

    l1mutable.add(l1mutable.remove(0));
    assertThat(l1mutable, is(resolver.getRotatedAddresses(TEST_HOST)));
  }

  @Test
  public void nonRotateWithRotateIpAddresses() throws UnknownHostException {
    HostNameResolver resolver = new HostNameResolver();
    List<InetAddress> l1 = resolver.getAddresses(TEST_HOST);
    List<InetAddress> l2 = resolver.getRotatedAddresses(TEST_HOST);
    assertThat(l2, is(l1));
  }
}
