/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.service.http.impl.grizzly.memory.manager;

import static java.lang.Integer.getInteger;

import static org.glassfish.grizzly.ThreadCache.obtainIndex;
import static org.glassfish.grizzly.ThreadCache.putToCache;
import static org.glassfish.grizzly.ThreadCache.takeFromCache;
import static org.glassfish.grizzly.monitoring.MonitoringUtils.loadJmxObject;

import com.mulesoft.service.http.impl.service.memory.management.DefaultByteBufferProvider;
import org.mule.runtime.api.memory.provider.ByteBufferProvider;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.Charset;
import java.util.Arrays;

import org.glassfish.grizzly.Buffer;
import org.glassfish.grizzly.Cacheable;
import org.glassfish.grizzly.ThreadCache;
import org.glassfish.grizzly.memory.AbstractMemoryManager;
import org.glassfish.grizzly.memory.ByteBufferWrapper;
import org.glassfish.grizzly.memory.MemoryProbe;
import org.glassfish.grizzly.memory.ThreadLocalPool;
import org.glassfish.grizzly.memory.WrapperAware;
import org.glassfish.grizzly.monitoring.MonitoringConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A Grizzly {@link org.glassfish.grizzly.memory.MemoryManager} that behaves exactly like
 * {@link org.glassfish.grizzly.memory.HeapMemoryManager} but that delegates the retrieval of {@link ByteBuffer} and arrays to a
 * managed mule {@link ByteBufferProvider}.
 *
 * @since 1.5.0
 */
public class MuleGrizzlyHeapMemoryManager extends AbstractMemoryManager<HeapBuffer> implements WrapperAware {

  private static final Logger LOGGER = LoggerFactory.getLogger(MuleGrizzlyHeapMemoryManager.class);

  private static final ThreadCache.CachedTypeIndex<MuleGrizzlyHeapMemoryManager.TrimmableHeapBuffer> CACHE_IDX =
      obtainIndex(
                  MuleGrizzlyHeapMemoryManager.TrimmableHeapBuffer.class,
                  getInteger(MuleGrizzlyHeapMemoryManager.class.getName()
                      + ".thb-cache-size", 8));

  private static final ThreadCache.CachedTypeIndex<MuleGrizzlyHeapMemoryManager.RecyclableByteBufferWrapper> BBW_CACHE_IDX =
      obtainIndex(
                  MuleGrizzlyHeapMemoryManager.RecyclableByteBufferWrapper.class,
                  getInteger(MuleGrizzlyHeapMemoryManager.class.getName()
                      + ".rbbw-cache-size", 2));

  private static ByteBufferProvider byteBufferProvider = new DefaultByteBufferProvider();

  public MuleGrizzlyHeapMemoryManager() {
    super();
  }

  public MuleGrizzlyHeapMemoryManager(final int maxBufferSize) {
    super(maxBufferSize);
  }

  /**
   * Sets the {@link ByteBufferProvider} for the Grizzly Memory Management. This is needed because the memory management is set in
   * a static block so there is no way to provide a managed {@link ByteBufferProvider} without setting it as a static attribute.
   *
   * @see {@link org.glassfish.grizzly.memory.MemoryManagerInitializer#initManager()}
   *
   * @param byteBufferProvider
   */
  public static void setMemoryManagerByteBufferProvider(ByteBufferProvider byteBufferProvider) {
    MuleGrizzlyHeapMemoryManager.byteBufferProvider = byteBufferProvider;
  }

  // ---------------------------------------------- Methods from MemoryManager

  /**
   * {@inheritDoc}
   */
  @Override
  public HeapBuffer allocate(final int size) {
    return allocateHeapBuffer(size);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public HeapBuffer allocateAtLeast(final int size) {
    return allocateHeapBufferAtLeast(size);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public HeapBuffer reallocate(final HeapBuffer oldBuffer, final int newSize) {
    return reallocateHeapBuffer(oldBuffer, newSize);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void release(final HeapBuffer buffer) {
    releaseHeapBuffer(buffer);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean willAllocateDirect(int size) {
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MonitoringConfig<MemoryProbe> getMonitoringConfig() {
    return monitoringConfig;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ThreadLocalPool createThreadLocalPool() {
    return new HeapBufferThreadLocalPool(this);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected Object createJmxManagementObject() {
    return loadJmxObject("org.glassfish.grizzly.memory.jmx.HeapMemoryManager", this,
                         MuleGrizzlyHeapMemoryManager.class);
  }

  // ----------------------------------------------- Methods from WrapperAware

  /**
   * {@inheritDoc}
   */
  @Override
  public HeapBuffer wrap(final byte[] data) {
    return createTrimAwareBuffer(data, 0, data.length);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public HeapBuffer wrap(final byte[] data, final int offset, final int length) {
    return createTrimAwareBuffer(data, offset, length);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public HeapBuffer wrap(final String s) {
    return wrap(s, Charset.defaultCharset());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public HeapBuffer wrap(final String s, final Charset charset) {
    return wrap(s.getBytes(charset));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Buffer wrap(final ByteBuffer byteBuffer) {
    if (byteBuffer.hasArray()) {
      return wrap(byteBuffer.array(), byteBuffer.arrayOffset() + byteBuffer.position(), byteBuffer.remaining());
    } else {
      return createByteBufferWrapper(byteBuffer);
    }
  }

  // ------------------------------------------------------- Protected Methods

  /**
   * Creates a new HeapBuffer with a a specified size.
   *
   * @param size size of buffer created
   * @return
   */
  protected HeapBuffer allocateHeapBuffer(final int size) {
    if (size > maxBufferSize) {
      // Don't use pool
      return createTrimAwareBuffer(size);
    }

    final ThreadLocalPool<HeapBuffer> threadLocalCache = getHeapBufferThreadLocalPool();
    if (threadLocalCache != null) {
      final int remaining = threadLocalCache.remaining();

      if (remaining == 0 || remaining < size) {
        reallocatePoolBuffer();
      }

      return (HeapBuffer) allocateFromPool(threadLocalCache, size);
    } else {
      return createTrimAwareBuffer(size);
    }
  }

  protected HeapBuffer allocateHeapBufferAtLeast(final int size) {
    if (size > maxBufferSize) {
      // Don't use pool
      return createTrimAwareBuffer(size);
    }

    final ThreadLocalPool<HeapBuffer> threadLocalCache = getHeapBufferThreadLocalPool();
    if (threadLocalCache != null) {
      int remaining = threadLocalCache.remaining();

      if (remaining == 0 || remaining < size) {
        reallocatePoolBuffer();
        remaining = threadLocalCache.remaining();
      }

      return (HeapBuffer) allocateFromPool(threadLocalCache, remaining);
    } else {
      return createTrimAwareBuffer(size);
    }
  }

  protected HeapBuffer reallocateHeapBuffer(HeapBuffer oldHeapBuffer, int newSize) {
    if (oldHeapBuffer.capacity() >= newSize) {
      return oldHeapBuffer;
    }

    final ThreadLocalPool<HeapBuffer> memoryPool = getHeapBufferThreadLocalPool();
    if (memoryPool != null) {
      final HeapBuffer newBuffer = memoryPool.reallocate(oldHeapBuffer, newSize);

      if (newBuffer != null) {

        return newBuffer;
      }
    }

    final HeapBuffer newHeapBuffer = allocateHeapBuffer(newSize);
    oldHeapBuffer.flip();
    return newHeapBuffer.put(oldHeapBuffer);
  }

  protected final void releaseHeapBuffer(final HeapBuffer heapBuffer) {
    final ThreadLocalPool<HeapBuffer> memoryPool = getHeapBufferThreadLocalPool();
    if (memoryPool != null) {
      memoryPool.release(heapBuffer.clear());
    }

  }

  // --------------------------------------------------------- Private Methods

  private void reallocatePoolBuffer() {
    final byte[] heap = byteBufferProvider.getByteArray(maxBufferSize);

    final MuleGrizzlyHeapMemoryManager.HeapBufferThreadLocalPool threadLocalCache =
        getHeapBufferThreadLocalPool();
    if (threadLocalCache != null) {
      threadLocalCache.reset(heap, 0, maxBufferSize);
    }
  }

  MuleGrizzlyHeapMemoryManager.TrimmableHeapBuffer createTrimAwareBuffer(final int length) {
    final byte[] heap = byteBufferProvider.getByteArray(length);

    return createTrimAwareBuffer(heap, 0, length);
  }

  MuleGrizzlyHeapMemoryManager.TrimmableHeapBuffer createTrimAwareBuffer(final byte[] heap,
                                                                         final int offset,
                                                                         final int length) {

    final MuleGrizzlyHeapMemoryManager.TrimmableHeapBuffer buffer =
        takeFromCache(CACHE_IDX);
    if (buffer != null) {
      buffer.initialize(this, heap, offset, length);
      return buffer;
    }

    return new MuleGrizzlyHeapMemoryManager.TrimmableHeapBuffer(this, heap, offset,
                                                                length);
  }

  private ByteBufferWrapper createByteBufferWrapper(final ByteBuffer underlyingByteBuffer) {

    final MuleGrizzlyHeapMemoryManager.RecyclableByteBufferWrapper buffer =
        takeFromCache(BBW_CACHE_IDX);
    if (buffer != null) {
      buffer.initialize(underlyingByteBuffer);
      return buffer;
    }

    return new MuleGrizzlyHeapMemoryManager.RecyclableByteBufferWrapper(underlyingByteBuffer);
  }

  /**
   * Gets the thread local buffer pool If the pool is not an instance of HeapBufferThreadLocalPool then null is returned
   *
   * @return
   */
  @SuppressWarnings("unchecked")
  private static MuleGrizzlyHeapMemoryManager.HeapBufferThreadLocalPool getHeapBufferThreadLocalPool() {
    final ThreadLocalPool pool = getThreadLocalPool();
    return pool instanceof MuleGrizzlyHeapMemoryManager.HeapBufferThreadLocalPool
        ? (MuleGrizzlyHeapMemoryManager.HeapBufferThreadLocalPool) pool
        : null;
  }

  // ---------------------------------------------------------- Nested Classes

  /**
   * Information about thread associated memory pool.
   */
  private static final class HeapBufferThreadLocalPool implements ThreadLocalPool<HeapBuffer> {

    /**
     * Memory pool
     */
    private byte[] pool;

    private int leftPos;
    private int rightPos;

    private int start;
    private int end;

    private final ByteBuffer[] byteBufferCache;
    private int byteBufferCacheSize = 0;
    private final MuleGrizzlyHeapMemoryManager mm;

    public HeapBufferThreadLocalPool(final MuleGrizzlyHeapMemoryManager mm) {
      this(mm, 8);
    }

    public HeapBufferThreadLocalPool(final MuleGrizzlyHeapMemoryManager mm,
                                     final int maxByteBufferCacheSize) {
      byteBufferCache = new ByteBuffer[maxByteBufferCacheSize];
      this.mm = mm;
    }

    @Override
    public HeapBuffer allocate(final int size) {
      final HeapBuffer allocated = mm.createTrimAwareBuffer(pool, rightPos, size);
      if (byteBufferCacheSize > 0) {
        allocated.byteBuffer = byteBufferCache[--byteBufferCacheSize];
        byteBufferCache[byteBufferCacheSize] = null;
      }

      rightPos += size;
      return allocated;
    }

    @Override
    public HeapBuffer reallocate(final HeapBuffer heapBuffer, final int newSize) {
      final int diff;

      if (isLastAllocated(heapBuffer) && remaining() >= (diff = newSize - heapBuffer.cap)) {

        rightPos += diff;
        heapBuffer.cap = newSize;
        heapBuffer.lim = newSize;

        return heapBuffer;
      }

      return null;
    }

    @Override
    public boolean release(final HeapBuffer heapBuffer) {
      boolean canCacheByteBuffer = heapBuffer.byteBuffer != null && byteBufferCacheSize < byteBufferCache.length;

      final boolean result;

      if (isLastAllocated(heapBuffer)) {
        rightPos -= heapBuffer.cap;

        if (leftPos == rightPos) {
          leftPos = rightPos = start;
        }

        result = true;
      } else if (isReleasableLeft(heapBuffer)) {
        leftPos += heapBuffer.cap;
        if (leftPos == rightPos) {
          leftPos = rightPos = start;
        }

        result = true;
      } else if (wantReset(heapBuffer.cap)) {
        reset(heapBuffer);

        result = true;
      } else {
        canCacheByteBuffer = canCacheByteBuffer && pool == heapBuffer.heap;
        result = false;
      }

      if (canCacheByteBuffer) {
        byteBufferCache[byteBufferCacheSize++] = heapBuffer.byteBuffer;
      }

      return result;
    }

    @Override
    public void reset(final HeapBuffer heapBuffer) {
      reset(heapBuffer.heap, heapBuffer.offset, heapBuffer.cap);
    }

    public void reset(final byte[] heap, final int offset, final int capacity) {
      if (pool != heap) {
        clearByteBufferCache();
        pool = heap;
      }

      leftPos = rightPos = start = offset;
      end = offset + capacity;
    }

    @Override
    public boolean wantReset(final int size) {
      return size - remaining() > 1024;
    }

    @Override
    public boolean isLastAllocated(final HeapBuffer oldHeapBuffer) {
      return oldHeapBuffer.heap == pool && oldHeapBuffer.offset + oldHeapBuffer.cap == rightPos;
    }

    private boolean isReleasableLeft(final HeapBuffer oldHeapBuffer) {
      return oldHeapBuffer.heap == pool && oldHeapBuffer.offset == leftPos;
    }

    @Override
    public HeapBuffer reduceLastAllocated(final HeapBuffer heapBuffer) {
      final int newPos = heapBuffer.offset + heapBuffer.cap;

      rightPos = newPos;

      return null;
    }

    @Override
    public int remaining() {
      return end - rightPos;
    }

    @Override
    public boolean hasRemaining() {
      return rightPos < end;
    }

    @Override
    public String toString() {
      return "(pool=" + pool.length + " pos=" + rightPos + " cap=" + end + ')';
    }

    private void clearByteBufferCache() {
      Arrays.fill(byteBufferCache, 0, byteBufferCacheSize, null);
      byteBufferCacheSize = 0;
    }

  } // END ByteBufferThreadLocalPool

  /**
   * {@link HeapBuffer} implementation, which supports trimming. In other words it's possible to return unused
   * {@link org.glassfish.grizzly.Buffer} space to pool.
   */
  private static final class TrimmableHeapBuffer extends HeapBuffer implements TrimAware {

    private MuleGrizzlyHeapMemoryManager mm;

    private TrimmableHeapBuffer(final MuleGrizzlyHeapMemoryManager mm, byte[] heap, int offset, int capacity) {
      super(heap, offset, capacity);
      this.mm = mm;
    }

    @Override
    public void trim() {
      checkDispose();

      final int sizeToReturn = cap - pos;

      if (sizeToReturn > 0) {
        final MuleGrizzlyHeapMemoryManager.HeapBufferThreadLocalPool threadLocalCache =
            getHeapBufferThreadLocalPool();
        if (threadLocalCache != null) {

          if (threadLocalCache.isLastAllocated(this)) {
            flip();
            cap = lim;
            threadLocalCache.reduceLastAllocated(this);

            return;
          } else if (threadLocalCache.wantReset(sizeToReturn)) {
            flip();

            cap = lim;

            threadLocalCache.reset(heap, offset + cap, sizeToReturn);
            return;
          }
        }
      }

      super.trim();
    }

    @Override
    public void recycle() {
      allowBufferDispose = false;

      putToCache(CACHE_IDX, this);
    }

    @Override
    public void dispose() {
      prepareDispose();
      mm.release(this);
      mm = null;

      byteBuffer = null;
      heap = null;
      pos = 0;
      offset = 0;
      lim = 0;
      cap = 0;
      order = ByteOrder.BIG_ENDIAN;
      bigEndian = true;
      recycle();
    }

    @Override
    protected HeapBuffer createHeapBuffer(final int offs, final int capacity) {
      return mm.createTrimAwareBuffer(heap, offs + offset, capacity);
    }

    void initialize(final MuleGrizzlyHeapMemoryManager mm, final byte[] heap,
                    final int offset, final int length) {

      this.mm = mm;
      this.heap = heap;
      this.offset = offset;
      pos = 0;
      cap = length;
      lim = length;

      disposeStackTrace = null;
    }
  } // END TrimAwareWrapper

  private final static class RecyclableByteBufferWrapper extends ByteBufferWrapper implements Cacheable {

    private RecyclableByteBufferWrapper(final ByteBuffer underlyingByteBuffer) {
      super(underlyingByteBuffer);
    }

    @Override
    public void recycle() {
      allowBufferDispose = false;

      putToCache(BBW_CACHE_IDX, this);
    }

    @Override
    public void dispose() {
      super.dispose();
      recycle();
    }

    private void initialize(final ByteBuffer underlyingByteBuffer) {
      visible = underlyingByteBuffer;
      disposeStackTrace = null;
    }

  }
}
