/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.service.http.impl.service;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * Dispatcher for the host name resolution.
 */
public class HostNameResolver {

  private ConcurrentMap<String, HostNameIpsRing> ipsForHostnames = new ConcurrentHashMap<>();

  private final AddressesRingFactory addressesRingFactory;

  /**
   * Builds a resolver with the default (ring) address factory.
   */
  public HostNameResolver() {
    this(new AddressesRingFactory() {

      @Override
      public HostNameIpsRing create(String hostName) {
        return new HostNameIpsRing(hostName);
      }
    });
  }

  /**
   * Builds a resolver with the given address factory.
   * 
   * @param addressesRingFactory the address factory to use
   */
  public HostNameResolver(AddressesRingFactory addressesRingFactory) {
    this.addressesRingFactory = addressesRingFactory;
  }

  private HostNameIpsRing getHostNameIpRing(String hostName) {
    // Avoid building too many discardable objects.
    if (!ipsForHostnames.containsKey(hostName)) {
      ipsForHostnames.putIfAbsent(hostName, getAddressesRingFactory().create(hostName));
    }
    return ipsForHostnames.get(hostName);
  }

  /**
   * Calls to this method are thread safe.
   *
   * @param hostName the hostName to resolve addresses for.
   * @return the available addresses for the given {@code hostname} this object was build with.
   * @throws UnknownHostException @{@link InetAddress#getAllByName(String)}
   */
  public List<InetAddress> getAddresses(String hostName) throws UnknownHostException {
    return getHostNameIpRing(hostName).getIpAddresses();
  }

  /**
   * Calls to this method are thread safe.
   * <p>
   * Two subsequent calls from the same thread may yield the same addresses in the same order.
   *
   * @param hostName the hostName to resolve addresses for.
   * @return the available addresses for the given {@code hostname} this object was build with.
   * @throws UnknownHostException @{@link InetAddress#getAllByName(String)}
   */
  public List<InetAddress> getRotatedAddresses(String hostName) throws UnknownHostException {
    return getHostNameIpRing(hostName).getRotatedIpAddresses();
  }

  /**
   * @return the factory for the actual host name resolvers.
   */
  public AddressesRingFactory getAddressesRingFactory() {
    return addressesRingFactory;
  }

  /**
   * Factory for the actual host name resolver.
   */
  public interface AddressesRingFactory {

    /**
     * Builds the actual host name resolver.
     */
    HostNameIpsRing create(String hostName);
  }
}
