/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.service.http.impl.service.client.builder;

import static com.ning.http.client.Realm.AuthScheme.NTLM;
import static org.mule.runtime.core.api.util.StringUtils.isEmpty;
import static org.mule.service.http.impl.service.client.GrizzlyHttpClient.HOST_SEPARATOR;

import com.ning.http.client.uri.Uri;
import org.mule.runtime.http.api.client.proxy.ProxyConfig;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;

import com.mulesoft.service.http.impl.service.HostNameResolver;
import com.ning.http.client.ProxyServer;
import com.ning.http.client.RequestBuilder;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.List;

/**
 * Handles the replacement of the address in the proxy configuration with a resolved ip address, leaving the request url as is.
 * <p>
 * The host name resolution for the request url host has to be performed by the proxy server.
 */
public class ProxyNameResolvingRequestBuilder extends NameResolvingRequestBuilder {

  private final ProxyConfig proxyConfig;

  public ProxyNameResolvingRequestBuilder(HttpRequest request, HostNameResolver hostNameResolver, ProxyConfig proxyConfig) {
    super(request, hostNameResolver);
    this.proxyConfig = proxyConfig;
  }

  @Override
  protected List<InetAddress> getAddresses() throws UnknownHostException {
    return getDomainNameResolver().getAddresses(proxyConfig.getHost());
  }

  @Override
  protected List<InetAddress> getRotateAddresses() throws UnknownHostException {
    return getDomainNameResolver().getRotatedAddresses(proxyConfig.getHost());
  }

  @Override
  public RequestBuilder setUrl(String url) {
    setProxyServer(buildProxy(proxyConfig));
    return super.setUrl(url);
  }

  @Override
  public RequestBuilder setUri(Uri uri) {
    setProxyServer(buildProxy(proxyConfig));
    return super.setUri(uri);
  }

  protected final ProxyServer buildProxy(ProxyConfig proxyConfig) {
    ProxyServer proxyServer;
    if (!isEmpty(proxyConfig.getUsername())) {
      proxyServer =
          new ProxyServer(proxyConfig.getHost(), proxyConfig.getPort(), proxyConfig.getUsername(), proxyConfig.getPassword());
      if (proxyConfig instanceof ProxyConfig.NtlmProxyConfig) {
        proxyServer.setNtlmDomain(((ProxyConfig.NtlmProxyConfig) proxyConfig).getNtlmDomain());
        try {
          proxyServer.setNtlmHost(InetAddress.getLocalHost().getHostName());
        } catch (UnknownHostException e) {
          // do nothing, let the default behaviour be used
        }
        proxyServer.setScheme(NTLM);
      }
    } else {
      proxyServer = new ProxyServer(proxyConfig.getHost(), proxyConfig.getPort());
    }

    if (proxyConfig.getNonProxyHosts() != null && !proxyConfig.getNonProxyHosts().isEmpty()) {
      for (final String host : proxyConfig.getNonProxyHosts().split(HOST_SEPARATOR)) {
        proxyServer.addNonProxyHost(host.trim());
      }
    }
    return proxyServer;
  }
}
