/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.service.http.impl.service;

import static com.mulesoft.service.http.impl.grizzly.memory.manager.MuleGrizzlyHeapMemoryManager.setMemoryManagerByteBufferProvider;
import static org.mule.runtime.api.memory.provider.type.ByteBufferType.HEAP;
import static org.mule.runtime.api.scheduler.SchedulerConfig.config;

import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.memory.management.MemoryManagementService;
import org.mule.runtime.api.memory.provider.ByteBufferProvider;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.ws.WebSocketBroadcaster;
import org.mule.service.http.impl.service.HttpServiceImplementation;
import org.mule.service.http.impl.service.client.HttpClientConnectionManager;
import org.mule.service.http.impl.service.server.HttpListenerConnectionManager;

import com.mulesoft.service.http.impl.service.client.EEHttpClientConnectionManager;
import com.mulesoft.service.http.impl.service.server.EEHttpListenerConnectionManager;
import com.mulesoft.service.http.impl.service.ws.GrizzlyWebSocketBroadcaster;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.inject.Inject;

/**
 * Implementation of {@link HttpService} that uses Grizzly to create {@link HttpServer}s and its Async HTTP Client provider to
 * create {@link org.mule.runtime.http.api.client.HttpClient}s, including a {@link HostNameResolver} to resolve host names.
 *
 * @since 1.0
 */
public class EEHttpServiceImplementation extends HttpServiceImplementation {

  private static final Logger LOGGER = LoggerFactory.getLogger(EEHttpServiceImplementation.class);

  public static final String HTTP_SERVICE_GRIZZLY_MEMORY_MANAGEMENT = "http.service.grizzly.memory.management";

  @Inject
  private MemoryManagementService memoryManagementService;
  private ByteBufferProvider byteBufferProvider;

  public EEHttpServiceImplementation(SchedulerService schedulerService) {
    super(schedulerService);
  }

  @Override
  protected HttpClientConnectionManager createClientConnectionManager() {
    return new EEHttpClientConnectionManager(schedulerService);
  }

  protected HttpListenerConnectionManager createListenerConnectionManager(SchedulerService schedulerService) {
    return new EEHttpListenerConnectionManager(schedulerService, config());
  }

  @Override
  public WebSocketBroadcaster newWebSocketBroadcaster() {
    return new GrizzlyWebSocketBroadcaster();
  }

  @Override
  public String getName() {
    return "HTTP EE Service";
  }

  @Override
  public void start() throws MuleException {
    MemoryManagementService memoryManagementServiceForInjectingByteBuffer = getMemoryManagementService();

    if (memoryManagementServiceForInjectingByteBuffer != null && byteBufferProvider == null) {
      byteBufferProvider = memoryManagementServiceForInjectingByteBuffer
          .getByteBufferProvider(HTTP_SERVICE_GRIZZLY_MEMORY_MANAGEMENT,
                                 HEAP);
      setMemoryManagerByteBufferProvider(byteBufferProvider);
    } else {
      LOGGER.warn(
                  "Memory Management is null. This is not a managed EEHttpServiceImpementation. A default Byte Buffer Provider will be used by the Mule Memory Manager.");
    }
    super.start();
  }

  @Override
  public void stop() throws MuleException {
    if (byteBufferProvider != null) {
      MemoryManagementService memoryManagementServiceForInjectingByteBuffer = getMemoryManagementService();
      memoryManagementServiceForInjectingByteBuffer.disposeByteBufferProvider(HTTP_SERVICE_GRIZZLY_MEMORY_MANAGEMENT);
      byteBufferProvider = null;
    }
    super.stop();
  }

  /**
   * @return the memory management service to use by the service
   */
  protected MemoryManagementService getMemoryManagementService() {
    return memoryManagementService;
  }
}
