/*
 * (c) 2003-2023 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.service.http.impl.service;

import static java.lang.Long.getLong;
import static java.lang.Math.abs;
import static java.lang.System.currentTimeMillis;
import static java.net.InetAddress.getAllByName;
import static java.util.Arrays.asList;
import static java.util.Arrays.copyOfRange;
import static java.util.Collections.singletonList;
import static org.apache.commons.lang3.ArrayUtils.addAll;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Object that provides resolved addresses for the host name it is built with.
 * <p>
 * The lookup itself is delegated to {@link InetAddress#getAllByName(String)}, which in turn delegates to the underlying platform.
 * <p>
 * Caching is already handled by the underlying mechanism and is controlled by system properties (See the javadoc for
 * {@link InetAddress}, <b>InetAddress Caching</b> section for insight on this mechanism).
 */
public class HostNameIpsRing {

  private static final Logger LOGGER = LoggerFactory.getLogger(HostNameIpsRing.class);

  private static final long cacheMaxTtl = getLong(HostNameIpsRing.class.getName() + ".cache.maxTtl", 1000L);
  private static final long cacheRefreshTime = getLong(HostNameIpsRing.class.getName() + ".cache.refreshTime", 100L);

  private final String hostName;
  private final AtomicInteger idxAddresses = new AtomicInteger(0);

  private InetAddress[] lastValues;
  private long lastRefreshTimestamp = -1;
  private Lock refreshLock = new ReentrantLock();

  /**
   * Builds the resolver for the given {@code hostName}.
   *
   * @param hostName the hostName to resolve addresses for.
   */
  public HostNameIpsRing(String hostName) {
    this.hostName = hostName;
  }

  private List<InetAddress> getIpAddressesRotatedFromRelativeIndex(int idx) throws UnknownHostException {
    final InetAddress[] addresses = getAddressesWithCache();
    int addressesCount = addresses.length;

    if (addressesCount == 1) {
      return singletonList(addresses[0]);
    } else {
      idx = abs(idx % addressesCount);

      List<InetAddress> addressesList = asList(addAll(copyOfRange(addresses, idx, addressesCount),
                                                      copyOfRange(addresses, 0, idx)));

      if (LOGGER.isTraceEnabled()) {
        LOGGER.trace("Host name {}: resolved ip addresses '{}' (base index {}).", hostName, addressesList.toString(), idx);
      }
      return addressesList;
    }
  }

  private InetAddress[] getAddressesWithCache() throws UnknownHostException {
    long currentTimeMillis = currentTimeMillis();

    if (currentTimeMillis - lastRefreshTimestamp > cacheMaxTtl) {
      refreshLock.lock();
      try {
        if (currentTimeMillis - lastRefreshTimestamp > cacheMaxTtl) {
          lastValues = getAllByName(hostName);
          lastRefreshTimestamp = currentTimeMillis;
        }
      } finally {
        refreshLock.unlock();
      }
    } else if (currentTimeMillis - lastRefreshTimestamp > cacheRefreshTime) {
      if (refreshLock.tryLock()) {
        try {
          lastValues = getAllByName(hostName);
          lastRefreshTimestamp = currentTimeMillis;
        } finally {
          refreshLock.unlock();
        }
      }
    }

    return lastValues;
  }

  /**
   * Calls to this method are thread safe.
   * <p>
   * Two subsequent calls from the same thread may yield the same address.
   *
   * @return the available addresses for the host name this object was build with, in the order they should be tried.
   * @throws UnknownHostException See {@link InetAddress#getAllByName(String)}
   */
  public List<InetAddress> getRotatedIpAddresses() throws UnknownHostException {
    return getIpAddressesRotatedFromRelativeIndex(idxAddresses.getAndIncrement());
  }

  /**
   * Calls to this method are thread safe. Subsequent calls to this method will would return the addresses in the exact same
   * order.
   *
   * @return the available addresses for the host name this object was build with, in the order they should be tried.
   * @throws UnknownHostException See {@link InetAddress#getAllByName(String)}
   */
  public List<InetAddress> getIpAddresses() throws UnknownHostException {
    return getIpAddressesRotatedFromRelativeIndex(idxAddresses.get());
  }
}
