/*
 * (c) 2003-2023 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.service.http.impl.service.ws;

import static java.lang.System.arraycopy;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.function.Function.identity;
import static org.glassfish.grizzly.utils.Futures.completable;

import org.mule.runtime.api.util.concurrent.Latch;
import org.mule.runtime.http.api.ws.WebSocket;
import org.mule.runtime.http.api.ws.exception.WebSocketConnectionException;

import java.io.InputStream;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;

import org.glassfish.grizzly.GrizzlyFuture;

/**
 * Utility methods for implementing WebSockets
 *
 * @since 1.3.0
 */
public final class WebSocketUtils {

  public static final int DEFAULT_DATA_FRAME_SIZE = 8 * 1024;

  public static CompletableFuture<Void> streamInDataFrames(InputStream content,
                                                           DataFrameEmitter emitter,
                                                           Function<Throwable, Throwable> exceptionMapper) {
    return streamInDataFrames(content, DEFAULT_DATA_FRAME_SIZE, emitter, exceptionMapper);
  }

  public static CompletableFuture<Void> streamInDataFrames(InputStream content,
                                                           int frameSize,
                                                           DataFrameEmitter emitter,
                                                           Function<Throwable, Throwable> exceptionMapper) {
    byte[] readBuffer = new byte[frameSize];
    byte[] writeBuffer = new byte[frameSize];
    int read;
    int write = 0;
    boolean streaming = false;
    CompletableFuture<Void> composedFuture = null;
    Latch latch = new Latch();
    AtomicReference<Throwable> error = new AtomicReference<>(null);

    try {
      while (error.get() == null && (read = content.read(readBuffer, 0, readBuffer.length)) != -1) {
        if (write > 0 && error.get() == null) {
          streaming = true;
          CompletableFuture frameFuture = emitter.stream(writeBuffer, 0, write, false).whenComplete((v, e) -> {
            if (e != null) {
              error.set(e);
              latch.release();
            }
          });
          if (composedFuture == null) {
            composedFuture = frameFuture;
          } else {
            composedFuture = composedFuture.thenCompose(v -> frameFuture);
          }
        }
        arraycopy(readBuffer, 0, writeBuffer, 0, read);
        write = read;
      }

      if (composedFuture != null) {
        composedFuture.whenComplete((v, e) -> latch.release());
        latch.await();
      }

      if (error.get() != null) {
        return failedFuture(exceptionMapper.apply(error.get()));
      }

      if (write == 0) {
        return completedFuture(null);
      }

      // because a bug in grizzly we need to create a byte array with the exact length
      if (write < writeBuffer.length) {
        byte[] exactSize = writeBuffer;
        writeBuffer = new byte[write];
        arraycopy(exactSize, 0, writeBuffer, 0, write);
      }

      if (error.get() != null) {
        return failedFuture(exceptionMapper.apply(error.get()));
      }

      if (streaming) {
        return emitter.stream(writeBuffer, 0, write, true);
      } else {
        return emitter.send(writeBuffer, 0, write);
      }
    } catch (Throwable t) {
      return failedFuture(exceptionMapper.apply(t));
    }
  }

  /**
   * Returns a void {@link CompletableFuture} that completes when the input {@code future} does. Exceptional completions are also
   * propagated using the same completion exception.
   *
   * @param future a {@link CompletableFuture} of a random type
   * @param <T>    the generic type of the original future.
   * @return a void {@link CompletableFuture}
   */
  public static <T> CompletableFuture<Void> asVoid(CompletableFuture<T> future) {
    return asVoid(future, identity());
  }

  /**
   * Returns a void {@link CompletableFuture} that completes when the input {@code future} does. Exceptional completions are also
   * propagated but mapped through the {@code exceptionMapper}
   *
   * @param future a {@link CompletableFuture} of a random type
   * @param <T>    the generic type of the original future.
   * @return a void {@link CompletableFuture}
   */
  public static <T> CompletableFuture<Void> asVoid(CompletableFuture<T> future,
                                                   Function<Throwable, Throwable> exceptionMapper) {

    CompletableFuture<Void> vf = new CompletableFuture<>();
    future.whenComplete((v, e) -> {
      if (e != null) {
        vf.completeExceptionally(exceptionMapper.apply(e));
      } else {
        vf.complete(null);
      }
    });

    return vf;
  }

  /**
   * Returns a void {@link CompletableFuture} that completes when the input {@code future} does. Exceptional completions are also
   * propagated using the same completion exception.
   *
   * @param future a {@link CompletableFuture} of a random type
   * @param <T>    the generic type of the original future.
   * @return a void {@link CompletableFuture}
   */
  public static <T> CompletableFuture<Void> asVoid(GrizzlyFuture<T> future) {
    return asVoid(future, identity());
  }

  /**
   * Returns a void {@link CompletableFuture} that completes when the input {@code future} does. Exceptional completions are also
   * propagated but mapped through the {@code exceptionMapper}
   *
   * @param future a {@link CompletableFuture} of a random type
   * @param <T>    the generic type of the original future.
   * @return a void {@link CompletableFuture}
   */
  public static <T> CompletableFuture<Void> asVoid(GrizzlyFuture<T> future, Function<Throwable, Throwable> exceptionMapper) {
    return asVoid(completable(future), exceptionMapper);
  }

  /**
   * Returns a {@link CompletableFuture} immediately and exceptionally completed with the given {@code t}
   * 
   * @param t   a {@link Throwable}
   * @param <T> the generic type of the future
   * @return a {@link CompletableFuture}
   */
  public static <T> CompletableFuture<T> failedFuture(Throwable t) {
    CompletableFuture<T> future = new CompletableFuture<>();
    future.completeExceptionally(t);

    return future;
  }

  /**
   * Maps Grizzly {@link Throwable} instances to Mule ones as defined in the Http Service API.
   *
   * @param t         the Throwable to be mapped
   * @param webSocket the {@link WebSocket} referred by the potentially mapped exception.
   * @return a mapped Throwable or the same input instance.
   */
  public static Throwable mapWsException(Throwable t, WebSocket webSocket) {
    if (t instanceof RuntimeException) {
      if ("Socket is already closed.".equals(t.getMessage()) || "Socket is not connected.".equals(t.getMessage())) {
        return new WebSocketConnectionException(webSocket, t);
      }
    }

    return t;
  }
}
