/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.service.oauth.internal.platform;

import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

import org.mule.runtime.api.el.MuleExpressionLanguage;
import org.mule.runtime.api.lock.LockFactory;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.proxy.ProxyConfig;
import org.mule.runtime.oauth.api.OAuthService;
import org.mule.runtime.oauth.api.PlatformManagedOAuthDancer;
import org.mule.runtime.oauth.api.builder.OAuthPlatformManagedDancerBuilder;
import org.mule.runtime.oauth.api.listener.PlatformManagedOAuthStateListener;
import org.mule.runtime.oauth.api.state.ResourceOwnerOAuthContext;
import org.mule.service.oauth.internal.builder.AbstractOAuthDancerBuilder;

import com.mulesoft.service.oauth.internal.platform.OCSClient.OCSClientFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.github.benmanes.caffeine.cache.LoadingCache;

/**
 * Default implementation of {@link OAuthPlatformManagedDancerBuilder}
 *
 * @since 1.0
 */
public class DefaultOAuthPlatformManagedDancerBuilder extends AbstractOAuthDancerBuilder<PlatformManagedOAuthDancer>
    implements OAuthPlatformManagedDancerBuilder {

  private final List<PlatformManagedOAuthStateListener> listeners = new ArrayList<>(2);
  private final OAuthService oauthService;
  private final OCSClientFactory ocsClientFactory;

  private String connectionUri;
  private String organizationId;
  private String platformUrl;

  public DefaultOAuthPlatformManagedDancerBuilder(OAuthService oauthService,
                                                  OCSClientFactory ocsClientFactory,
                                                  SchedulerService schedulerService,
                                                  LockFactory lockProvider,
                                                  Map<String, ResourceOwnerOAuthContext> tokensStore,
                                                  LoadingCache<Pair<TlsContextFactory, ProxyConfig>, HttpClient> httpClientCache,
                                                  MuleExpressionLanguage expressionEvaluator) {
    super(schedulerService, lockProvider, tokensStore, httpClientCache, expressionEvaluator);
    this.ocsClientFactory = ocsClientFactory;
    this.oauthService = oauthService;
  }

  @Override
  public OAuthPlatformManagedDancerBuilder connectionUri(String connectionUri) {
    this.connectionUri = connectionUri;
    return this;
  }

  @Override
  public OAuthPlatformManagedDancerBuilder organizationId(String organizationId) {
    this.organizationId = organizationId;
    return this;
  }

  @Override
  public OAuthPlatformManagedDancerBuilder platformUrl(String platformUrl) {
    this.platformUrl = platformUrl;
    return this;
  }

  @Override
  public OAuthPlatformManagedDancerBuilder addListener(PlatformManagedOAuthStateListener listener) {
    requireNonNull(listener, "Cannot add a null listener");
    listeners.add(listener);

    return this;
  }

  @Override
  public PlatformManagedOAuthDancer build() {
    checkArgument(isNotBlank(clientId), "clientId cannot be blank");
    checkArgument(isNotBlank(clientSecret), "clientSecret cannot be blank");
    checkArgument(isNotBlank(tokenUrl), "tokenUrl cannot be blank");
    checkArgument(isNotBlank(platformUrl), "platformUrl cannot be blank");
    checkArgument(isNotBlank(connectionUri), "connectionUri cannot be blank");
    checkArgument(isNotBlank(organizationId), "organizationId cannot be blank");

    return new DefaultPlatformManagedDancer(name,
                                            connectionUri,
                                            organizationId,
                                            platformUrl,
                                            ocsClientFactory,
                                            oauthService,
                                            clientId,
                                            clientSecret,
                                            tokenUrl,
                                            scopes,
                                            clientCredentialsLocation,
                                            encoding,
                                            responseAccessTokenExpr,
                                            responseRefreshTokenExpr,
                                            responseExpiresInExpr,
                                            customParametersExtractorsExprs,
                                            resourceOwnerIdTransformer,
                                            schedulerService,
                                            lockProvider,
                                            tokensStore,
                                            httpClientFactory.get(),
                                            expressionEvaluator,
                                            listeners);
  }
}
