/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package com.mulesoft.service.oauth.internal.platform;

import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.runtime.api.util.Preconditions.checkArgument;

import com.mulesoft.service.oauth.internal.platform.config.DefaultPlatformManagedDancerConfig;

import org.mule.oauth.client.api.http.HttpClientFactory;
import org.mule.oauth.client.api.state.ResourceOwnerOAuthContext;
import org.mule.oauth.client.internal.builder.AbstractOAuthDancerBuilder;
import org.mule.oauth.client.internal.config.AuthDancerCredentialConfig;
import org.mule.oauth.client.internal.config.TokenParserConfig;
import org.mule.runtime.api.el.MuleExpressionLanguage;
import org.mule.runtime.api.lock.LockFactory;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.oauth.api.OAuthService;
import org.mule.runtime.oauth.api.PlatformManagedOAuthDancer;
import org.mule.runtime.oauth.api.builder.OAuthPlatformManagedDancerBuilder;
import org.mule.runtime.oauth.api.listener.PlatformManagedOAuthStateListener;

import com.mulesoft.service.oauth.internal.platform.OCSClient.OCSClientFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Default implementation of {@link OAuthPlatformManagedDancerBuilder}
 *
 * @since 1.0
 */
public class DefaultOAuthPlatformManagedDancerBuilder extends AbstractOAuthDancerBuilder<PlatformManagedOAuthDancer>
    implements OAuthPlatformManagedDancerBuilder {

  private final List<PlatformManagedOAuthStateListener> listeners = new ArrayList<>(2);
  private final OAuthService oauthService;
  private final OCSClientFactory ocsClientFactory;

  private String connectionUri;
  private String organizationId;
  private String platformUrl;
  private String apiVersion;

  public DefaultOAuthPlatformManagedDancerBuilder(OAuthService oauthService,
                                                  OCSClientFactory ocsClientFactory,
                                                  SchedulerService schedulerService,
                                                  LockFactory lockProvider,
                                                  Map<String, ResourceOwnerOAuthContext> tokensStore,
                                                  HttpClientFactory httpClientFactory,
                                                  MuleExpressionLanguage expressionEvaluator) {
    super(schedulerService, lockProvider, tokensStore, httpClientFactory, expressionEvaluator);
    this.ocsClientFactory = ocsClientFactory;
    this.oauthService = oauthService;
  }

  @Override
  public OAuthPlatformManagedDancerBuilder connectionUri(String connectionUri) {
    this.connectionUri = connectionUri;
    return this;
  }

  @Override
  public OAuthPlatformManagedDancerBuilder organizationId(String organizationId) {
    this.organizationId = organizationId;
    return this;
  }

  @Override
  public OAuthPlatformManagedDancerBuilder platformUrl(String platformUrl) {
    this.platformUrl = platformUrl;
    return this;
  }

  @Override
  public OAuthPlatformManagedDancerBuilder apiVersion(String apiVersion) {
    this.apiVersion = apiVersion;
    return this;
  }


  @Override
  public OAuthPlatformManagedDancerBuilder addListener(PlatformManagedOAuthStateListener listener) {
    requireNonNull(listener, "Cannot add a null listener");
    listeners.add(listener);

    return this;
  }

  @Override
  public PlatformManagedOAuthDancer build() {
    checkArgument(isNotBlank(clientId), "clientId cannot be blank");
    checkArgument(isNotBlank(clientSecret), "clientSecret cannot be blank");
    checkArgument(isNotBlank(tokenUrl), "tokenUrl cannot be blank");
    checkArgument(isNotBlank(platformUrl), "platformUrl cannot be blank");
    checkArgument(isNotBlank(connectionUri), "connectionUri cannot be blank");
    checkArgument(isNotBlank(organizationId), "organizationId cannot be blank");

    DefaultPlatformManagedDancerConfig config = new DefaultPlatformManagedDancerConfig();
    config.setName(name);

    config.setConnectionUri(connectionUri);
    config.setOrganizationId(organizationId);

    config.setPlatformUrl(platformUrl);
    config.setApiVersion(apiVersion);
    config.setOcsClientFactory(ocsClientFactory);
    config.setOauthService(oauthService);
    config.setCredentialConfig(AuthDancerCredentialConfig.builder()
        .withClientId(clientId)
        .withClientSecret(clientSecret)
        .withClientCredentialsLocation(clientCredentialsLocation)
        .build());
    config.setTokenUrl(tokenUrl);
    config.setScopes(scopes);
    config.setEncoding(encoding);
    config.setTokenParserConfig(TokenParserConfig.builder()
        .withResponseAccessTokenExpr(responseAccessTokenExpr)
        .withResponseRefreshTokenExpr(responseRefreshTokenExpr)
        .withResponseExpiresInExpr(responseExpiresInExpr)
        .build());
    config.setCustomParametersExtractorsExprs(customParametersExtractorsExprs);
    config.setResourceOwnerIdTransformer(resourceOwnerIdTransformer);
    config.setSchedulerService(schedulerService);
    config.setLockProvider(lockProvider);
    config.setTokensStore(tokensStore);
    config.setHttpClient(httpClientFactory.get());
    config.setExpressionEvaluator(expressionEvaluator);
    config.setListeners(listeners);

    return new DefaultPlatformManagedDancer(config);
  }
}
