/*
 * (c) 2003-2019 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.test.extension.dsl.ee;

import static com.google.common.base.Preconditions.checkArgument;
import static java.lang.String.format;
import static java.util.Collections.singletonList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.mule.runtime.api.component.ComponentIdentifier.builder;
import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.CONNECTION;
import static org.mule.runtime.app.declaration.api.fluent.ElementDeclarer.newArtifact;
import static org.mule.runtime.app.declaration.api.fluent.ElementDeclarer.newListValue;
import static org.mule.runtime.app.declaration.api.fluent.ElementDeclarer.newObjectValue;
import static org.mule.runtime.app.declaration.api.fluent.ElementDeclarer.newParameterGroup;
import static org.mule.runtime.extension.api.ExtensionConstants.STREAMING_STRATEGY_PARAMETER_NAME;
import static org.mule.runtime.extension.api.ExtensionConstants.TLS_PARAMETER_NAME;
import static org.mule.runtime.extension.api.declaration.type.StreamingStrategyTypeBuilder.REPEATABLE_FILE_STORE_BYTES_STREAM_ALIAS;
import static org.mule.runtime.internal.dsl.DslConstants.CORE_PREFIX;
import static org.mule.runtime.internal.dsl.DslConstants.EE_PREFIX;
import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.runtime.app.declaration.api.ElementDeclaration;
import org.mule.runtime.app.declaration.api.fluent.ElementDeclarer;
import org.mule.runtime.dsl.api.xml.parser.XmlConfigurationDocumentLoader;
import org.mule.runtime.config.api.dsl.model.DslElementModel;
import org.mule.runtime.config.api.dsl.model.DslElementModelFactory;
import org.mule.runtime.config.api.dsl.processor.ArtifactConfig;
import org.mule.runtime.config.api.dsl.processor.xml.XmlApplicationServiceRegistry;
import org.mule.runtime.config.internal.ModuleDelegatingEntityResolver;
import org.mule.runtime.config.internal.model.ApplicationModel;
import org.mule.runtime.config.internal.model.ComponentModel;
import org.mule.runtime.core.api.extension.MuleExtensionModelProvider;
import org.mule.runtime.core.api.registry.SpiServiceRegistry;
import org.mule.runtime.core.api.util.xmlsecurity.XMLSecureFactories;
import org.mule.runtime.dsl.api.component.config.ComponentConfiguration;
import org.mule.runtime.dsl.api.xml.XmlNamespaceInfoProvider;
import org.mule.runtime.dsl.api.xml.parser.ConfigFile;
import org.mule.runtime.dsl.api.xml.parser.ConfigLine;
import org.mule.runtime.dsl.internal.xml.parser.XmlApplicationParser;
import org.mule.runtime.config.internal.dsl.xml.XmlNamespaceInfoProviderSupplier;

import com.google.common.collect.ImmutableSet;
import com.mulesoft.mule.runtime.core.api.extension.MuleEeExtensionModelProvider;

import java.io.InputStream;
import java.io.StringWriter;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.junit.Before;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

public abstract class AbstractElementModelTestCase extends MuleArtifactFunctionalTestCase {

  protected static final String HTTP_LISTENER_CONFIG = "httpListener";
  protected static final String COMPONENTS_FLOW = "testFlow";
  protected static final int LISTENER_PATH = 0;
  protected static final int LOGGER_PATH = 1;
  protected static final int TRANSFORM_PATH = 2;

  protected DslResolvingContext dslContext;
  protected DslElementModelFactory modelResolver;
  protected ApplicationModel applicationModel;
  protected Document doc;

  @Before
  public void setup() throws Exception {
    Set<ExtensionModel> extensions = muleContext.getExtensionManager().getExtensions();

    dslContext = DslResolvingContext.getDefault(ImmutableSet.<ExtensionModel>builder()
        .addAll(extensions).add(MuleExtensionModelProvider.getExtensionModel())
        .add(MuleEeExtensionModelProvider.getExtensionModel()).build());
    modelResolver = DslElementModelFactory.getDefault(dslContext);
  }

  @Override
  protected boolean isDisposeContextPerClass() {
    return true;
  }

  protected <T extends NamedObject> DslElementModel<T> resolve(ComponentConfiguration component) {
    Optional<DslElementModel<T>> elementModel = modelResolver.create(component);
    assertThat(elementModel.isPresent(), is(true));
    return elementModel.get();
  }

  protected <T extends NamedObject> DslElementModel<T> resolve(ElementDeclaration component) {
    Optional<DslElementModel<T>> elementModel = modelResolver.create(component);
    assertThat(elementModel.isPresent(), is(true));
    return elementModel.get();
  }

  protected ComponentConfiguration getAppElement(ApplicationModel applicationModel, String name) {
    Optional<ComponentModel> component = applicationModel.findTopLevelNamedComponent(name);
    assertThat(component.isPresent(), is(true));
    return component.get().getConfiguration();
  }

  protected <T> DslElementModel<T> getChild(DslElementModel<? extends NamedObject> parent, ComponentConfiguration component) {
    return getChild(parent, component.getIdentifier());
  }

  protected <T> DslElementModel<T> getChild(DslElementModel<? extends NamedObject> parent,
                                            ComponentIdentifier identifier) {
    Optional<DslElementModel<T>> elementModel = parent.findElement(identifier);
    assertThat(format("Failed fetching child '%s' from parent '%s'", identifier.getName(),
                      parent.getModel().getName()),
               elementModel.isPresent(), is(true));
    return elementModel.get();
  }

  protected <T> DslElementModel<T> getChild(DslElementModel<? extends NamedObject> parent,
                                            String name) {
    Optional<DslElementModel<T>> elementModel = parent.findElement(name);
    assertThat(format("Failed fetching child '%s' from parent '%s'", name,
                      parent.getModel().getName()),
               elementModel.isPresent(), is(true));
    return elementModel.get();
  }

  private <T> DslElementModel<T> getAttribute(DslElementModel<? extends NamedObject> parent, String component) {
    Optional<DslElementModel<T>> elementModel = parent.findElement(component);
    assertThat(format("Failed fetching attribute '%s' from parent '%s'", component, parent.getModel().getName()),
               elementModel.isPresent(), is(true));
    return elementModel.get();
  }

  protected ComponentIdentifier newIdentifier(String name, String ns) {
    return builder().name(name).namespace(ns).build();
  }

  protected void assertHasParameter(ParameterizedModel model, String name) {
    assertThat(model.getAllParameterModels()
        .stream().anyMatch(p -> p.getName().equals(name)), is(true));
  }

  protected void assertHasParameterGroup(ParameterizedModel model, String name) {
    assertThat(model.getParameterGroupModels()
        .stream().anyMatch(p -> p.getName().equals(name)), is(true));
  }

  protected void assertAttributeIsPresent(DslElementModel<? extends ParameterizedModel> element, String name) {
    assertHasParameter(element.getModel(), name);
    DslElementModel<NamedObject> databaseParam = getAttribute(element, name);
    assertThat(databaseParam.getDsl().supportsAttributeDeclaration(), is(true));
    assertThat(databaseParam.getDsl().supportsChildDeclaration(), is(false));
  }

  protected void assertElementName(DslElementModel propertiesElement, String name) {
    assertThat(propertiesElement.getDsl().getElementName(), is(name));
  }

  // Scaffolding
  protected ApplicationModel loadApplicationModel() throws Exception {
    return loadApplicationModel(getConfigFile());
  }

  protected ApplicationModel loadApplicationModel(String configFile) throws Exception {
    InputStream appIs = Thread.currentThread().getContextClassLoader().getResourceAsStream(configFile);
    checkArgument(appIs != null, "The given application was not found as resource");

    Document document = XmlConfigurationDocumentLoader.noValidationDocumentLoader()
        .loadDocument(() -> XMLSecureFactories.createDefault().getSAXParserFactory(),
                      new ModuleDelegatingEntityResolver(muleContext.getExtensionManager().getExtensions()), configFile, appIs);

    XmlApplicationServiceRegistry customRegistry = new XmlApplicationServiceRegistry(new SpiServiceRegistry(), dslContext);
    ConfigLine configLine =
        new XmlApplicationParser(XmlNamespaceInfoProviderSupplier
            .createFromPluginClassloaders(cl -> customRegistry.lookupProviders(XmlNamespaceInfoProvider.class, cl).stream()
                .collect(Collectors.toList()), singletonList(AbstractElementModelTestCase.class.getClassLoader())))
                    .parse(document.getDocumentElement())
                    .orElseThrow(() -> new Exception("Failed to load config"));

    ArtifactConfig artifactConfig = new ArtifactConfig.Builder()
        .addConfigFile(new ConfigFile(configFile, singletonList(configLine)))
        .build();

    return new ApplicationModel(artifactConfig, new ArtifactDeclaration(),
                                uri -> muleContext.getExecutionClassLoader().getResourceAsStream(uri));
  }

  protected String write() throws Exception {
    // write the content into xml file
    TransformerFactory transformerFactory = TransformerFactory.newInstance();

    Transformer transformer = transformerFactory.newTransformer();
    transformer.setOutputProperty(OutputKeys.INDENT, "yes");
    transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");

    DOMSource source = new DOMSource(doc);
    StringWriter writer = new StringWriter();
    transformer.transform(source, new StreamResult(writer));

    return writer.getBuffer().toString().replaceAll("\n|\r", "");
  }

  protected void createAppDocument() throws ParserConfigurationException {
    DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
    factory.setNamespaceAware(true);
    DocumentBuilder docBuilder = factory.newDocumentBuilder();

    this.doc = docBuilder.newDocument();
    Element mule = doc.createElement("mule");
    doc.appendChild(mule);
    mule.setAttributeNS("http://www.w3.org/2000/xmlns/", "xmlns", "http://www.mulesoft.org/schema/mule/core");
    mule.setAttributeNS("http://www.w3.org/2001/XMLSchema-instance",
                        "xsi:schemaLocation", getExpectedSchemaLocation());
  }

  protected String getExpectedSchemaLocation() {
    return "http://www.mulesoft.org/schema/mule/core http://www.mulesoft.org/schema/mule/core/current/mule.xsd";
  }

  protected ArtifactDeclaration createAppDeclarationWithTransform() {

    ElementDeclarer http = ElementDeclarer.forExtension("HTTP");
    ElementDeclarer core = ElementDeclarer.forExtension(CORE_PREFIX);
    ElementDeclarer ee = ElementDeclarer.forExtension(EE_PREFIX);

    return newArtifact()
        .withGlobalElement(
                           http.newConfiguration("listenerConfig")
                               .withRefName("httpListener")
                               .withParameterGroup(newParameterGroup()
                                   .withParameter("basePath", "/")
                                   .getDeclaration())
                               .withConnection(http.newConnection("listener")
                                   .withParameterGroup(newParameterGroup()
                                       .withParameter(TLS_PARAMETER_NAME, newObjectValue()
                                           .withParameter("key-store", newObjectValue()
                                               .withParameter("path", "ssltest-keystore.jks")
                                               .withParameter("password", "changeit")
                                               .withParameter("keyPassword", "changeit")
                                               .build())
                                           .build())
                                       .getDeclaration())
                                   .withParameterGroup(newParameterGroup(CONNECTION)
                                       .withParameter("host", "localhost")
                                       .withParameter("port", "49019")
                                       .withParameter("protocol", "HTTPS")
                                       .getDeclaration())
                                   .getDeclaration())
                               .getDeclaration())
        .withGlobalElement(core.newConstruct("flow").withRefName("testFlow")
            .withParameterGroup(newParameterGroup()
                .withParameter("initialState", "stopped")
                .getDeclaration())
            .withComponent(http.newSource("listener")
                .withConfig("httpListener")
                .withParameterGroup(newParameterGroup()
                    .withParameter("path", "testBuilder")
                    .withParameter("responseStreamingMode", "AUTO")
                    .withParameter(STREAMING_STRATEGY_PARAMETER_NAME,
                                   newObjectValue()
                                       .ofType(
                                               REPEATABLE_FILE_STORE_BYTES_STREAM_ALIAS)
                                       .withParameter("bufferUnit", "KB")
                                       .withParameter("inMemorySize", "2048")
                                       .build())
                    .getDeclaration())
                .getDeclaration())
            .withComponent(core.newOperation("logger")
                .withParameterGroup(newParameterGroup()
                    .withParameter("message", "#[payload]")
                    .getDeclaration())
                .getDeclaration())
            .withComponent(ee.newOperation("transform")
                .withParameterGroup(newParameterGroup("Message")
                    .withParameter("setPayload", newObjectValue()
                        .withParameter("script",
                                       "<![CDATA["
                                           + "output application/java\n"
                                           + "                    ---\n"
                                           + "                    payload\n"
                                           + "                    ]]>")
                        .build())
                    .withParameter("setAttributes", newObjectValue()
                        .withParameter("resource", "myWeaveResource.dw")
                        .build())
                    .getDeclaration())
                .withParameterGroup(newParameterGroup("Set Variables")
                    .withParameter("setVariables",
                                   newListValue()
                                       .withValue(newObjectValue()
                                           .withParameter("variableName", "bar")
                                           .withParameter("script",
                                                          "<![CDATA["
                                                              + "output application/java\n"
                                                              + "                    ---\n"
                                                              + "                    null\n"
                                                              + "                    ]]>")
                                           .build())
                                       .withValue(newObjectValue()
                                           .withParameter("variableName", "foo")
                                           .withParameter("resource", "myWeaveResource.dw")
                                           .build())
                                       .build())
                    .getDeclaration())
                .getDeclaration())
            .getDeclaration())
        .getDeclaration();

  }
}
