/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.cache;

import static com.mulesoft.test.allure.AllureConstants.CacheFeature.CACHE;
import static java.lang.Runtime.getRuntime;
import static java.lang.Thread.currentThread;
import static java.util.concurrent.TimeUnit.SECONDS;

import org.mule.runtime.api.exception.DefaultMuleException;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.scheduler.SchedulerConfig;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.core.api.processor.Processor;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;

import javax.inject.Inject;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import io.qameta.allure.Feature;

@Feature(CACHE)
public class NonBlockingCacheTestCase extends AbstractCacheFunctionalTestCase {

  private static CountDownLatch enterLatch;
  private static CountDownLatch exitLatch;
  private static List<String> threadNames;

  @Inject
  private SchedulerService schedulerService;

  private Scheduler cpuLightScheduler;
  private Scheduler testScheduler;

  @Override
  protected String getConfigFile() {
    return "non-blocking-cache-test-config.xml";
  }

  @Before
  public void before() {
    enterLatch = new CountDownLatch(1);
    exitLatch = new CountDownLatch(1);
    threadNames = new ArrayList<>();

    cpuLightScheduler = schedulerService.cpuLightScheduler();
    testScheduler = schedulerService.customScheduler(SchedulerConfig.config().withName("test")
        .withMaxConcurrentTasks(2 + getRuntime().availableProcessors() * 2));
  }

  @After
  public void after() {
    testScheduler.stop();
    cpuLightScheduler.stop();
  }

  @Test
  public void synchronizedCache() throws Exception {
    flowRunner("synchronizedCache").withPayload(TEST_MESSAGE).run();
    flowRunner("synchronizedCache").withPayload(TEST_MESSAGE).run();
  }

  @Test
  public void cacheSynchronizeFreesCpuLight() throws Exception {
    testScheduler.submit(() -> flowRunner("cacheSynchronizeFreesCpuLight").withPayload(TEST_MESSAGE).run());
    enterLatch.await();

    for (int i = 0; i < (getRuntime().availableProcessors() * 2) - 1; ++i) {
      testScheduler.submit(() -> flowRunner("cacheSynchronizeFreesCpuLight").withPayload(TEST_MESSAGE).run());
    }

    Thread.sleep(2000);

    // Verify cpu light still accepts tasks
    cpuLightScheduler.submit(() -> {
    }).get(5, SECONDS);
  }

  @Test
  public void unsynchronizedCache() throws Exception {
    flowRunner("unsynchronizedCache").withPayload(TEST_MESSAGE).run();
    flowRunner("unsynchronizedCache").withPayload(TEST_MESSAGE).run();
  }

  public static final class LatchedProcessor implements Processor {

    @Override
    public CoreEvent process(CoreEvent event) throws MuleException {
      enterLatch.countDown();

      threadNames.add(currentThread().getName());

      try {
        exitLatch.await();
      } catch (InterruptedException e) {
        currentThread().interrupt();
        throw new DefaultMuleException(e);
      }
      return event;
    }

  }
}

