/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.cache.config;

import static java.util.Arrays.asList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.mule.runtime.core.api.util.StreamingUtils.asCursorProvider;
import org.mule.functional.api.flow.FlowRunner;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.api.streaming.object.CursorIteratorProvider;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.core.api.util.IOUtils;
import org.mule.runtime.core.api.util.func.CheckedRunnable;

import com.mulesoft.mule.runtime.cache.api.key.MuleEventKeyGenerator;
import com.mulesoft.mule.runtime.cache.api.response.ResponseGenerator;
import com.mulesoft.mule.test.cache.AbstractCacheFunctionalTestCase;

import java.io.Serializable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.junit.Test;

public class CacheConfigTestCase extends AbstractCacheFunctionalTestCase {

  private static final String GENERATED_KEY = "theKey";
  private static final String GENERATED_RESPONSE = "theResponse";
  private static final String OBJECT_STORE_ID = "objectStore";
  private static final List<String> STREAM_OBJECTS = asList("Apple", "Banana", "Kiwi");

  @Override
  protected String getConfigFile() {
    return "config/cache-config.xml";
  }

  // TODO MULE-17934 remove this
  @Override
  protected boolean isGracefulShutdown() {
    return true;
  }

  @Test
  public void testMessageProcessorDefaultConfig() throws Exception {
    String flowName = "CacheRouterDefault";
    CoreEvent firstResponse = runCachedFlow(flowName);
    CoreEvent secondResponse = runCachedFlow(flowName);

    assertThat(firstResponse.getMessage().getPayload().getValue(), equalTo(0));
    assertThat(secondResponse.getMessage().getPayload().getValue(), equalTo(0));
  }

  @Test
  public void testMessageProcessorFilterExpressionConfig() throws Exception {
    String flowName = "CacheRouterWithFilterExpression";
    CoreEvent firstResponse = runCachedFlow(flowName);
    CoreEvent secondResponse = runCachedFlow(flowName);

    // Filters skips the cache, so same request gets different outputs
    assertThat(firstResponse.getMessage().getPayload().getValue(), equalTo(0));
    assertThat(secondResponse.getMessage().getPayload().getValue(), equalTo(1));
  }

  @Test
  public void testMessageProcessorCachingStrategyConfig() throws Exception {
    String flowName = "CacheRouterWithGlobalObjectStore";
    String secondFlowName = "CacheRouterWithGlobalObjectStore2";
    CoreEvent firstResponse = runCachedFlow(flowName);
    CoreEvent secondResponse = runCachedFlow(flowName);
    CoreEvent thirdResponse = runCachedFlow(secondFlowName);
    assertThat(firstResponse.getMessage().getPayload().getValue(), equalTo(0));
    assertThat(secondResponse.getMessage().getPayload().getValue(), equalTo(0));
    assertThat(thirdResponse.getMessage().getPayload().getValue(), equalTo(0));

    // Confirms that the configured object store was used
    ObjectStore<Serializable> objectStore = registry.<ObjectStore>lookupByName(OBJECT_STORE_ID).get();
    assertThat(objectStore.allKeys(), hasSize(1));
  }

  @Test
  public void testMessageProcessorPrivateCachingStrategyConfig() throws Exception {
    String flowName = "CacheRouterWithPrivateObjectStore";
    CoreEvent firstResponse = runCachedFlow(flowName);
    CoreEvent secondResponse = runCachedFlow(flowName);
    assertThat(firstResponse.getMessage().getPayload().getValue(), equalTo(0));
    assertThat(secondResponse.getMessage().getPayload().getValue(), equalTo(0));
  }

  @Test
  public void testMessageProcessorKeyExpressionConfig() throws Exception {
    String flowName = "CacheRouterWithKeyGenerationExpression";
    CoreEvent firstResponse = runCachedFlow(flowName);
    CoreEvent secondResponse = runCachedFlow(flowName);

    // Filters skips the cache, so same request gets different outputs
    assertThat(firstResponse.getMessage().getPayload().getValue(), equalTo(0));
    assertThat(secondResponse.getMessage().getPayload().getValue(), equalTo(0));

    // Confirms that the configured object store was used
    ObjectStore objectStore = registry.<ObjectStore>lookupByName(OBJECT_STORE_ID).get();
    assertThat(objectStore.contains(TEST_PAYLOAD), equalTo(true));
  }

  @Test
  public void testMessageProcessorKeyGeneratorConfig() throws Exception {
    String flowName = "CacheRouterWithKeyGenerator";
    CoreEvent firstResponse = runCachedFlow(flowName);
    CoreEvent secondResponse = runCachedFlow(flowName);

    // Filters skips the cache, so same request gets different outputs
    assertThat(firstResponse.getMessage().getPayload().getValue(), equalTo(0));
    assertThat(secondResponse.getMessage().getPayload().getValue(), equalTo(0));

    // Confirms that the configured object store was used
    ObjectStore objectStore = registry.<ObjectStore>lookupByName(OBJECT_STORE_ID).get();
    assertThat(objectStore.contains(GENERATED_KEY), equalTo(true));
  }

  @Test
  public void testMessageProcessorResponseGeneratorConfig() throws Exception {
    String flowName = "CacheRouterWithResponseGenerator";
    CoreEvent firstResponse = runCachedFlow(flowName);
    CoreEvent secondResponse = runCachedFlow(flowName);

    // Filters skips the cache, so same request gets different outputs
    assertThat(firstResponse.getMessage().getPayload().getValue(), equalTo(0));
    assertThat(secondResponse.getMessage().getPayload().getValue(), equalTo(GENERATED_RESPONSE));
  }

  @Test
  public void messageWithRepeatableInputStream() throws Exception {
    doTestRepeatableInputStream("CacheRepeatableStream");
  }

  @Test
  public void persistentMessageWithRepeatableInputStream() throws Exception {
    doTestRepeatableInputStream("CachePersistentRepeatableStream");
  }

  private void doTestRepeatableInputStream(String flowName) throws Exception {
    doMultiple(3, () -> {
      final String streamKey = "stream";
      final String counter = "counter";

      final CoreEvent response = runStreamingCachedFlow(flowName);
      final Map<String, Object> payload = (Map<String, Object>) response.getMessage().getPayload().getValue();
      final CursorStreamProvider stream = (CursorStreamProvider) payload.get(streamKey);

      assertThat(IOUtils.toString(stream), equalTo(TEST_PAYLOAD));
      assertThat(payload.get(counter), equalTo(0));
    });
  }

  private void doMultiple(int count, CheckedRunnable task) {
    for (int i = 0; i < count; i++) {
      task.run();
    }
  }

  @Test
  public void messageWithRepeatableIterator() throws Exception {
    doTestRepeatableIteratorStream("CacheRepeatableStream");
  }

  @Test
  public void persistentMessageWithRepeatableIterator() throws Exception {
    doTestRepeatableIteratorStream("CachePersistentRepeatableStream");
  }

  private void doTestRepeatableIteratorStream(String flowName) {
    final String streamKey = "stream";
    final String counter = "counter";
    doMultiple(3, () -> {

      final CoreEvent response = runObjectStreamingCacheFow(flowName);
      final Map<String, Object> payload = (Map<String, Object>) response.getMessage().getPayload().getValue();
      final CursorIteratorProvider provider = (CursorIteratorProvider) payload.get(streamKey);
      final Iterator<String> stream = (Iterator<String>) provider.openCursor();
      final List<String> list = new LinkedList<>();

      stream.forEachRemaining(list::add);

      assertThat(list, equalTo(STREAM_OBJECTS));
      assertThat(payload.get(counter), equalTo(0));
    });
  }

  private CoreEvent runStreamingCachedFlow(String flowName) throws Exception {
    return flowRunner(flowName)
        .withVariable("stream", asCursorProvider(TEST_PAYLOAD.getBytes()))
        .run();
  }

  private CoreEvent runObjectStreamingCacheFow(String flowName) throws Exception {
    return flowRunner(flowName)
        .withVariable("stream", asCursorProvider(STREAM_OBJECTS))
        .run();
  }

  private CoreEvent runCachedFlow(String flowName) throws Exception {
    return new FlowRunner(registry, flowName).withPayload(TEST_PAYLOAD).run();
  }

  public static class TestMuleEventKeyGenerator implements MuleEventKeyGenerator {

    public TestMuleEventKeyGenerator() {

    }

    @Override
    public String generateKey(CoreEvent event) {
      return GENERATED_KEY;
    }
  }


  public static class TestResponseGenerator implements ResponseGenerator {

    @Override
    public CoreEvent create(CoreEvent request, CoreEvent cachedResponse) {
      return CoreEvent.builder(request).message(Message.of(GENERATED_RESPONSE)).build();
    }
  }
}
