/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.cache.integration;

import static com.mulesoft.test.allure.AllureConstants.CacheFeature.CACHE;
import static junit.framework.Assert.assertNotNull;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.junit.Assert.assertThat;

import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.serialization.ObjectSerializer;
import org.mule.runtime.api.store.ObjectStoreException;
import org.mule.runtime.api.store.SimpleMemoryObjectStore;

import com.mulesoft.mule.test.cache.AbstractCacheFunctionalTestCase;

import java.io.Serializable;

import javax.inject.Inject;

import org.junit.Test;

import io.qameta.allure.Feature;

/**
 * Test for a flow that uses a cache and a non-blocking processing strategy. A custom object store (see below) has been defined in
 * order to simulate what would happen in a cluster environment where the event has to be serialized in order to be store in the
 * manager-object-store.
 */
@Feature(CACHE)
public class CacheNonBlockingInFlowTestCase extends AbstractCacheFunctionalTestCase {

  @Override
  protected String getConfigFile() {
    return "integration/cache-nonblocking-in-flow-config.xml";
  }

  @Test
  public void usingCacheSynchronizeOnNonBlockingFlowTest() throws Exception {
    doTestCacheNonBlockingFlow("cacheSynchronizeOnNonBlockingFlow");
  }

  @Test
  public void usingCacheNonSynchronizeOnNonBlockingFlowTest() throws Exception {
    doTestCacheNonBlockingFlow("cacheNonSynchronizeOnNonBlockingFlow");
  }

  private void doTestCacheNonBlockingFlow(String flowName) throws Exception {
    Message response1 = flowRunner(flowName).withVariable("requestId", "1").withPayload(TEST_PAYLOAD).run().getMessage();
    assertNotNull(response1);

    Message response2 = flowRunner(flowName).withVariable("requestId", "1").withPayload(TEST_PAYLOAD).run().getMessage();
    assertNotNull(response2);

    assertThat(response2.getPayload().getValue(), equalTo(response1.getPayload().getValue()));
  }

  public static class CustomObjectStore extends SimpleMemoryObjectStore<Serializable> {

    @Inject
    private ObjectSerializer objectSerializer;

    @Override
    protected void doStore(String id, Serializable value) throws ObjectStoreException {
      byte[] serialized = objectSerializer.getInternalProtocol().serialize(value);
      super.doStore(id, serialized);
    }

    @Override
    protected Serializable doRetrieve(String key) throws ObjectStoreException {
      Serializable retrieved = super.doRetrieve(key);
      if (retrieved != null) {
        retrieved = objectSerializer.getInternalProtocol().deserialize((byte[]) retrieved);
      }
      return retrieved;
    }
  }
}
