/*
 * (c) 2003-2023 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.domain.service.scheduler;

import static org.mule.runtime.core.api.config.MuleProperties.OBJECT_CLASSLOADER_REPOSITORY;
import static org.mule.test.allure.AllureConstants.SchedulerServiceFeature.SCHEDULER_SERVICE;

import static java.lang.Thread.currentThread;
import static java.util.Arrays.asList;
import static java.util.Collections.singletonMap;
import static java.util.concurrent.TimeUnit.SECONDS;

import static com.google.common.collect.ImmutableSet.of;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.sameInstance;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;

import org.mule.functional.junit4.DomainFunctionalTestCase;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.scheduler.SchedulerBusyException;
import org.mule.runtime.api.scheduler.SchedulerContainerPoolsConfig;
import org.mule.runtime.api.scheduler.SchedulerPoolsConfigFactory;
import org.mule.runtime.api.util.concurrent.Latch;
import org.mule.runtime.core.api.extension.MuleExtensionModelProvider;
import org.mule.runtime.core.internal.config.preferred.PreferredObjectSelector;
import org.mule.runtime.module.artifact.api.classloader.ClassLoaderRepository;

import com.mulesoft.mule.runtime.core.api.extension.MuleEeExtensionModelProvider;
import com.mulesoft.mule.runtime.core.internal.config.scheduler.SchedulerPoolsArtifactConfig;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.function.Consumer;
import java.util.function.Function;

import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameter;
import org.junit.runners.Parameterized.Parameters;

import org.hamcrest.Matcher;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;

@Feature(SCHEDULER_SERVICE)
@RunWith(Parameterized.class)
public class SchedulerConfigurationTestCase extends DomainFunctionalTestCase {

  private static final String DOMAIN_WITH_CONFIG = "service/scheduler/scheduler-config-domain-shared.xml";
  private static final String DOMAIN_EMPTY = "service/scheduler/empty-domain.xml";
  private static final String APP_WITH_CONFIG = "service/scheduler/scheduler-config-app.xml";
  private static final String APP_EMPTY = "service/scheduler/empty-app.xml";

  private static final String APP_1_NAME = "app1";
  private static final String APP_2_NAME = "app2";

  @Parameter(0)
  public String domainConfig;
  @Parameter(1)
  public ApplicationConfig[] appConfigs;
  @Parameter(2)
  public Function<SchedulerPoolsConfigFactory, Matcher<SchedulerPoolsConfigFactory>> configMatcherFactory;
  @Parameter(3)
  public Consumer<ExpectedException> expectedConfigurer;

  @Parameters
  public static List<Object[]> parameters() {
    return asList(new Object[][] {
        parametersSet(DOMAIN_WITH_CONFIG, new ApplicationConfig[] {
            new ApplicationConfig(APP_1_NAME, APP_EMPTY),
            new ApplicationConfig(APP_2_NAME, APP_EMPTY)},
                      cfg2 -> allOf(instanceOf(SchedulerPoolsArtifactConfig.class), sameInstance(cfg2)),
                      e -> e.expectCause(instanceOf(SchedulerBusyException.class))),
        parametersSet(DOMAIN_WITH_CONFIG, new ApplicationConfig[] {
            new ApplicationConfig(APP_1_NAME, APP_EMPTY),
            new ApplicationConfig(APP_2_NAME, APP_WITH_CONFIG)},
                      cfg2 -> allOf(instanceOf(SchedulerPoolsArtifactConfig.class), not(sameInstance(cfg2))),
                      e -> {
                      }),
        parametersSet(DOMAIN_EMPTY, new ApplicationConfig[] {
            new ApplicationConfig(APP_1_NAME, APP_EMPTY),
            new ApplicationConfig(APP_2_NAME, APP_WITH_CONFIG)},
                      cfg2 -> allOf(instanceOf(SchedulerContainerPoolsConfig.class), not(sameInstance(cfg2))),
                      e -> {
                      }),
        parametersSet(DOMAIN_EMPTY, new ApplicationConfig[] {
            new ApplicationConfig(APP_1_NAME, APP_WITH_CONFIG),
            new ApplicationConfig(APP_2_NAME, APP_EMPTY)},
                      cfg2 -> allOf(instanceOf(SchedulerPoolsArtifactConfig.class), not(sameInstance(cfg2))),
                      e -> {
                      }),
        parametersSet(DOMAIN_EMPTY, new ApplicationConfig[] {
            new ApplicationConfig(APP_1_NAME, APP_WITH_CONFIG),
            new ApplicationConfig(APP_2_NAME, APP_WITH_CONFIG)},
                      cfg2 -> allOf(instanceOf(SchedulerPoolsArtifactConfig.class), not(sameInstance(cfg2))),
                      e -> {
                      })
    });
  }

  private static Object[] parametersSet(String domainConfig, ApplicationConfig[] appConfigs,
                                        Function<SchedulerPoolsConfigFactory, Matcher<SchedulerPoolsConfigFactory>> configMatcherFactory,
                                        Consumer<ExpectedException> expectedConfigurer) {
    return new Object[] {domainConfig, appConfigs, configMatcherFactory, expectedConfigurer};
  }

  @Rule
  public ExpectedException expected = none();

  @Override
  protected String getDomainConfig() {
    return domainConfig;
  }

  @Override
  public ApplicationConfig[] getConfigResources() {
    return appConfigs;
  }

  @Override
  protected Map<String, Object> getDomainStartUpRegistryObjects() {
    // Verify inconsistency between mockito and bytebuddy for using a mock here
    return singletonMap(OBJECT_CLASSLOADER_REPOSITORY, new ClassLoaderRepository() {

      @Override
      public Optional<String> getId(ClassLoader classLoader) {
        return null;
      }

      @Override
      public Optional<ClassLoader> find(String classLoaderId) {
        return null;
      }
    });
  }

  @Test
  public void configInContext() {
    final SchedulerPoolsConfigFactory app1SchedulerPoolsConfig =
        new PreferredObjectSelector<SchedulerPoolsConfigFactory>()
            .select(getInfrastructureForApp(APP_1_NAME).getRegistry().lookupAllByType(SchedulerPoolsConfigFactory.class)
                .iterator());

    final SchedulerPoolsConfigFactory app2SchedulerPoolsConfig =
        new PreferredObjectSelector<SchedulerPoolsConfigFactory>()
            .select(getInfrastructureForApp(APP_2_NAME).getRegistry().lookupAllByType(SchedulerPoolsConfigFactory.class)
                .iterator());

    assertThat(app1SchedulerPoolsConfig, getMatcher(app2SchedulerPoolsConfig));
  }

  private Matcher<SchedulerPoolsConfigFactory> getMatcher(SchedulerPoolsConfigFactory target) {
    return configMatcherFactory.apply(target);
  }

  @Test
  @Ignore("MULE-12310 - Check that the schedulers returned are not the unit test ones")
  @Issue("MULE-12310")
  public void appUsesCustomPools() throws InterruptedException, ExecutionException {
    Latch latch = new Latch();
    final Scheduler scheduler1Outer = getInfrastructureForApp(APP_1_NAME).getSchedulerService().cpuLightScheduler();
    final Scheduler scheduler1Inner = getInfrastructureForApp(APP_1_NAME).getSchedulerService().ioScheduler();

    final Scheduler scheduler2Outer = getInfrastructureForApp(APP_2_NAME).getSchedulerService().cpuLightScheduler();
    final Scheduler scheduler2Inner = getInfrastructureForApp(APP_2_NAME).getSchedulerService().ioScheduler();

    executeWork(latch, scheduler1Outer, scheduler1Inner);

    expectedConfigurer.accept(expected);

    // since IO pool max size is just one, we expect the second submission to be rejected
    try {
      executeWork(latch, scheduler2Outer, scheduler2Inner);
    } finally {
      latch.countDown();
      scheduler1Outer.shutdownNow();
      scheduler1Inner.shutdownNow();
      scheduler2Outer.shutdownNow();
      scheduler2Inner.shutdownNow();
    }
  }

  private void executeWork(Latch latch, final Scheduler schedulerOuter, final Scheduler schedulerInner)
      throws InterruptedException, ExecutionException {
    final Future<?> submit = schedulerOuter.submit(() -> {
      schedulerInner.submit(() -> {
        try {
          latch.await(DEFAULT_TEST_TIMEOUT_SECS, SECONDS);
        } catch (InterruptedException e) {
          currentThread().interrupt();
        }
      });
      return null;
    });

    submit.get();
  }

  @Override
  protected Set<ExtensionModel> getExtensionModels() {
    return of(MuleExtensionModelProvider.getExtensionModel(), MuleEeExtensionModelProvider.getExtensionModel());
  }
}
