/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.test.module.http.functional;

import static com.mulesoft.mule.runtime.core.internal.config.EEMuleProperties.KRYO_OBJECT_SERIALIZER;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonMap;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.withSettings;
import static org.mule.runtime.api.util.MultiMap.emptyMultiMap;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.startIfNeeded;
import org.mule.extension.http.api.HttpRequestAttributes;
import org.mule.extension.http.api.HttpRequestAttributesBuilder;
import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.api.serialization.ObjectSerializer;
import org.mule.runtime.api.serialization.SerializationProtocol;
import org.mule.runtime.core.api.config.ConfigurationBuilder;
import org.mule.runtime.http.api.HttpService;
import org.mule.service.http.impl.service.HttpServiceImplementation;
import org.mule.tck.SimpleUnitTestSupportSchedulerService;

import com.vdurmont.semver4j.Semver;

import java.security.cert.Certificate;
import java.util.List;
import java.util.Map;

import javax.inject.Inject;
import javax.inject.Named;

import io.qameta.allure.Description;
import org.junit.BeforeClass;
import org.junit.Test;


public class HttpRequestAttributesKryoSerializationTestCase extends MuleArtifactFunctionalTestCase {

  private SchedulerService schedulerService = new SimpleUnitTestSupportSchedulerService();
  private HttpService httpService = new HttpServiceImplementation(schedulerService);

  // Kryo serialization of attributes will correctly work from 4.1.5. If using a previous runtime version and an http-connector
  // version from 1.4.0,
  // the client certificate needs to be get first to correctly serialize it.
  private boolean isFeatureSupported = new Semver(getMavenProjectVersionProperty()).isGreaterThan(new Semver("4.1.4"));

  @Inject
  @Named(KRYO_OBJECT_SERIALIZER)
  private ObjectSerializer kryoObjectSerializer;

  private static final String CERTIFICATE = "TEST_CERTIFICATE";

  private static Certificate certificateMock;

  private HttpRequestAttributesBuilder baseBuilder =
      new HttpRequestAttributesBuilder()
          .listenerPath("/listener/path")
          .relativePath("/relative/path")
          .version("1.0")
          .scheme("scheme")
          .method("GET")
          .requestPath("/request/path")
          .remoteAddress("http://10.1.2.5:8080/")
          .localAddress("http://127.0.0.1:8080/")
          .requestUri("http://127.0.0.1/gateway")
          .headers(emptyMultiMap())
          .queryString("queryParam1=queryParam1&queryParam2=queryParam2")
          .queryParams(emptyMultiMap())
          .uriParams(emptyMap());


  @BeforeClass
  public static void setup() {
    certificateMock = mock(Certificate.class, withSettings().serializable());
    when(certificateMock.toString()).thenReturn(CERTIFICATE);
  }

  @Override
  protected boolean doTestClassInjection() {
    return true;
  }

  @Override
  protected void addBuilders(List<ConfigurationBuilder> builders) {
    super.addBuilders(builders);
    try {
      startIfNeeded(httpService);
    } catch (MuleException e) {
      // do nothing
    }
  }

  @Override
  protected Map<String, Object> getStartUpRegistryObjects() {
    return singletonMap(httpService.getName(), httpService);
  }

  @Override
  protected String[] getConfigFiles() {
    return new String[] {};
  }

  @Test
  @Description("Validate that Kryo can correctly serialize HttpRequestAttributes with lazy Certificate. This only works because the tests are ran with the latest runtime. These tests will fail from 4.1.1 to 4.1.4")
  public void withLazyResolvedCertificate() throws Exception {
    HttpRequestAttributes attributes = baseBuilder.clientCertificate(() -> certificateMock).build();
    assertSerialization(attributes);
  }

  @Test
  @Description("Validate that Kryo can correctly serialize HttpRequestAttributes with explicit Certificate. This only works because the tests are ran with the latest runtime. These tests will fail from 4.1.1 to 4.1.4")
  public void withExplicitCertificate() throws Exception {
    HttpRequestAttributes attributes = baseBuilder.clientCertificate(certificateMock).build();
    assertSerialization(attributes);
  }

  private HttpRequestAttributes serializeAndDeserialize(HttpRequestAttributes attributes) {
    SerializationProtocol protocol = kryoObjectSerializer.getInternalProtocol();
    return protocol.deserialize(protocol.serialize(attributes));
  }

  private void assertSerialization(HttpRequestAttributes originalAttributes) {

    // This is needed because if running in runtimes older than to 4.1.5, Kryo will lose the clientCertificate if
    // serializing the HttpRequestAttributes, unless obtained first. This is to avoid getting test failures when running
    // in those runtimes.
    if (!isFeatureSupported) {
      originalAttributes.getClientCertificate();
    }

    HttpRequestAttributes processedAttributes = serializeAndDeserialize(originalAttributes);

    assertThat(processedAttributes.getListenerPath(), equalTo(originalAttributes.getListenerPath()));
    assertThat(processedAttributes.getRelativePath(), equalTo(originalAttributes.getRelativePath()));
    assertThat(processedAttributes.getVersion(), equalTo(originalAttributes.getVersion()));
    assertThat(processedAttributes.getScheme(), equalTo(originalAttributes.getScheme()));
    assertThat(processedAttributes.getMethod(), equalTo(originalAttributes.getMethod()));
    assertThat(processedAttributes.getRequestPath(), equalTo(originalAttributes.getRequestPath()));
    assertThat(processedAttributes.getRemoteAddress(), equalTo(originalAttributes.getRemoteAddress()));
    assertThat(processedAttributes.getLocalAddress(), equalTo(originalAttributes.getLocalAddress()));
    assertThat(processedAttributes.getRequestUri(), equalTo(originalAttributes.getRequestUri()));
    assertThat(processedAttributes.getHeaders(), equalTo(originalAttributes.getHeaders()));
    assertThat(processedAttributes.getQueryString(), equalTo(originalAttributes.getQueryString()));
    assertThat(processedAttributes.getQueryParams(), equalTo(originalAttributes.getQueryParams()));
    assertThat(processedAttributes.getUriParams(), equalTo(originalAttributes.getUriParams()));
    assertThat(processedAttributes.getMaskedRequestPath(), equalTo(originalAttributes.getMaskedRequestPath()));
    assertThat(processedAttributes.getClientCertificate().toString(),
               is(equalTo(originalAttributes.getClientCertificate().toString())));
  }
}
