/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.test.module.http.functional;

import static java.net.InetAddress.getByName;
import static org.hamcrest.Matchers.instanceOf;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mule.runtime.api.exception.ExceptionHelper.getRootException;
import org.mule.tck.junit4.rule.DynamicPort;

import com.mulesoft.service.http.impl.service.HostNameResolver;
import com.mulesoft.service.http.impl.service.client.EEHttpClientConnectionManager;

import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.List;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.mockito.Mock;
import org.mockito.Mockito;

@Feature("HTTP Connector")
@Story("DNS lookup")
public class NameResolvingGrizzlyHttpClientTestCase extends AbstractHttpFunctionalTestCase {

  protected static final String RESOLVED_PROXY_HOST_NAME = "resolvedProxyHost";
  protected static final String RESOLVED_HOST_NAME = "resolvedHost";

  private static InetAddress connectableAddress;
  private static InetAddress notConnectableAddress;

  private static List<InetAddress> allFailingAddresses;
  private static List<InetAddress> allConnectableAddresses;
  private static List<InetAddress> connectableOnRetryAddresses;
  private static List<InetAddress> connectableOnMultipleRetryAddresses;

  @Rule
  public DynamicPort proxyPort = new DynamicPort("proxyPort");
  @Rule
  public DynamicPort httpPort = new DynamicPort("httpPort");

  @Mock
  protected HostNameResolver hostNameResolver = mock(HostNameResolver.class);

  @Override
  protected String getConfigFile() {
    return "name-resolving-grizzly-http-client.xml";
  }

  @BeforeClass
  public static void beforeClass() throws UnknownHostException {
    connectableAddress = getByName("127.0.0.1");
    notConnectableAddress = getByName("203.0.113.1");

    allFailingAddresses = Arrays.asList(notConnectableAddress, notConnectableAddress);
    connectableOnRetryAddresses = Arrays.asList(notConnectableAddress, connectableAddress);
    allConnectableAddresses = Arrays.asList(connectableAddress);
    connectableOnMultipleRetryAddresses = Arrays.asList(notConnectableAddress, notConnectableAddress, connectableAddress);
  }

  @Override
  protected void doSetUpBeforeMuleContextCreation() throws Exception {
    super.doSetUpBeforeMuleContextCreation();
    EEHttpClientConnectionManager.hostNameResolver = hostNameResolver;
  }

  @Override
  protected void doTearDownAfterMuleContextDispose() throws Exception {
    super.doTearDownAfterMuleContextDispose();
    EEHttpClientConnectionManager.hostNameResolver = new HostNameResolver();
  }

  protected void validateRetryNoProxy() throws UnknownHostException {
    verify(hostNameResolver).getRotatedAddresses(RESOLVED_HOST_NAME);
    verify(hostNameResolver, Mockito.never()).getAddresses(RESOLVED_HOST_NAME);
    verify(hostNameResolver, Mockito.never()).getRotatedAddresses(RESOLVED_PROXY_HOST_NAME);
    verify(hostNameResolver, Mockito.never()).getAddresses(RESOLVED_PROXY_HOST_NAME);
  }

  protected void validateRetryWithProxy() throws UnknownHostException {
    verify(hostNameResolver).getRotatedAddresses(RESOLVED_HOST_NAME);
    verify(hostNameResolver).getRotatedAddresses(RESOLVED_PROXY_HOST_NAME);
    verify(hostNameResolver, Mockito.never()).getAddresses(RESOLVED_PROXY_HOST_NAME);
    verify(hostNameResolver, Mockito.never()).getAddresses(RESOLVED_PROXY_HOST_NAME);
  }

  protected void validateRetryProxyOnly() throws UnknownHostException {
    verify(hostNameResolver, Mockito.never()).getRotatedAddresses(RESOLVED_HOST_NAME);
    verify(hostNameResolver, Mockito.never()).getAddresses(RESOLVED_HOST_NAME);
    verify(hostNameResolver).getRotatedAddresses(RESOLVED_PROXY_HOST_NAME);
    verify(hostNameResolver, Mockito.never()).getAddresses(RESOLVED_PROXY_HOST_NAME);
  }

  @Test
  public void retriesThenConnects() throws Exception {
    when(hostNameResolver.getRotatedAddresses(RESOLVED_HOST_NAME)).thenReturn(connectableOnRetryAddresses);
    when(hostNameResolver.getAddresses(RESOLVED_HOST_NAME)).thenReturn(connectableOnRetryAddresses);

    flowRunner("outer").runAndVerify("inner");
    validateRetryNoProxy();
  }

  @Test
  public void retriesMultipleRetriesThenConnects() throws Exception {
    when(hostNameResolver.getRotatedAddresses(RESOLVED_HOST_NAME)).thenReturn(connectableOnMultipleRetryAddresses);
    when(hostNameResolver.getAddresses(RESOLVED_HOST_NAME)).thenReturn(connectableOnMultipleRetryAddresses);

    flowRunner("outer").runAndVerify("inner");
    validateRetryNoProxy();
  }

  @Test
  public void retriesThenFails() throws Exception {
    when(hostNameResolver.getRotatedAddresses(RESOLVED_HOST_NAME)).thenReturn(allFailingAddresses);
    when(hostNameResolver.getAddresses(RESOLVED_HOST_NAME)).thenReturn(allFailingAddresses);

    Exception e = flowRunner("outer").runExpectingException();
    assertThat(getRootException(e), instanceOf(SocketException.class));
    validateRetryNoProxy();
  }

  @Test
  public void retriesThenConnectsHttpProxy() throws Exception {
    when(hostNameResolver.getRotatedAddresses(RESOLVED_PROXY_HOST_NAME)).thenReturn(connectableOnRetryAddresses);
    when(hostNameResolver.getRotatedAddresses(RESOLVED_HOST_NAME)).thenReturn(allConnectableAddresses);
    when(hostNameResolver.getAddresses(RESOLVED_PROXY_HOST_NAME)).thenReturn(connectableOnRetryAddresses);
    when(hostNameResolver.getAddresses(RESOLVED_HOST_NAME)).thenReturn(allConnectableAddresses);

    flowRunner("outerWithProxy").runAndVerify("inner");
    validateRetryWithProxy();
  }

  @Test
  public void retriesThenFailsHttpProxy() throws Exception {
    when(hostNameResolver.getRotatedAddresses(RESOLVED_PROXY_HOST_NAME)).thenReturn(allFailingAddresses);
    when(hostNameResolver.getRotatedAddresses(RESOLVED_HOST_NAME)).thenReturn(allConnectableAddresses);
    when(hostNameResolver.getAddresses(RESOLVED_PROXY_HOST_NAME)).thenReturn(allFailingAddresses);
    when(hostNameResolver.getAddresses(RESOLVED_HOST_NAME)).thenReturn(allConnectableAddresses);

    Exception e = flowRunner("outerWithProxy").runExpectingException();
    assertThat(getRootException(e), instanceOf(SocketException.class));
    validateRetryProxyOnly();
  }

}
