/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.test.module.http.functional;

import static java.net.InetAddress.getByName;
import static java.net.InetAddress.getLocalHost;
import static java.util.Arrays.asList;
import static org.hamcrest.Matchers.instanceOf;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.withSettings;
import static org.mule.functional.junit4.matchers.ThrowableCauseMatcher.hasCause;
import static org.mule.runtime.api.exception.ExceptionHelper.getRootException;
import org.mule.tck.junit4.rule.DynamicPort;

import com.mulesoft.service.http.impl.service.HostNameResolver;
import com.mulesoft.service.http.impl.service.client.EEHttpClientConnectionManager;

import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.List;

import javax.net.ssl.SSLHandshakeException;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

@Feature("HTTP Connector")
@Story("DNS lookup")
public class NameResolvingGrizzlyHttpClientTlsTestCase extends AbstractHttpFunctionalTestCase {

  private static final String RESOLVED_HOST_NAME = "resolvedHost";
  private static final String LOCALHOST_HOST_NAME = "localhost";

  private static InetAddress connectableAddress;
  private static InetAddress notConnectableAddress;
  private static InetAddress notSANAddress;

  private static List<InetAddress> allFailingAddresses;
  private static List<InetAddress> allConnectableAddresses;
  private static List<InetAddress> connectableOnRetryAddresses;

  @Rule
  public ExpectedException socketException = ExpectedException.none();

  @Rule
  public DynamicPort proxyPort = new DynamicPort("proxyPort");
  @Rule
  public DynamicPort httpPort = new DynamicPort("httpPort");
  @Rule
  public DynamicPort httpsPort = new DynamicPort("httpsPort");

  private HostNameResolver hostNameResolver = mock(HostNameResolver.class, withSettings().lenient());

  @BeforeClass
  public static void beforeClass() throws UnknownHostException {
    connectableAddress = getByName("127.0.0.1");
    notConnectableAddress = getByName("203.0.113.1");
    notSANAddress = getLocalHost();

    allFailingAddresses = asList(notConnectableAddress, notConnectableAddress);
    connectableOnRetryAddresses = asList(notConnectableAddress, connectableAddress);
    allConnectableAddresses = asList(connectableAddress);
  }

  @Override
  protected String getConfigFile() {
    return "name-resolving-grizzly-http-client-tls.xml";
  }

  @Override
  protected void doSetUpBeforeMuleContextCreation() throws Exception {
    super.doSetUpBeforeMuleContextCreation();
    EEHttpClientConnectionManager.hostNameResolver = hostNameResolver;
  }

  @Override
  protected void doTearDownAfterMuleContextDispose() throws Exception {
    super.doTearDownAfterMuleContextDispose();
    EEHttpClientConnectionManager.hostNameResolver = new HostNameResolver();
  }

  @Test
  public void retriesThenConnects() throws Exception {
    when(hostNameResolver.getRotatedAddresses(LOCALHOST_HOST_NAME)).thenReturn(connectableOnRetryAddresses);

    flowRunner("outer").runAndVerify("inner");
    verify(hostNameResolver).getRotatedAddresses(LOCALHOST_HOST_NAME);
  }

  @Test
  public void retriesThenFails() throws Exception {
    when(hostNameResolver.getRotatedAddresses(LOCALHOST_HOST_NAME)).thenReturn(allFailingAddresses);

    Exception e = flowRunner("outer").runExpectingException();
    assertThat(getRootException(e), instanceOf(SocketException.class));
    verify(hostNameResolver).getRotatedAddresses(LOCALHOST_HOST_NAME);
  }

  /**
   * SSL host name verification is done by comparing the SANs of the server certificate to the host in the url, regardless of the
   * actual address of the underlying socket.
   */
  @Test
  public void validHostInUrlInvalidInSocket() throws Exception {
    when(hostNameResolver.getRotatedAddresses(LOCALHOST_HOST_NAME)).thenReturn(asList(notSANAddress));

    flowRunner("outer").runAndVerify("inner");
    verify(hostNameResolver).getRotatedAddresses(LOCALHOST_HOST_NAME);
  }

  @Test
  public void invalidHostInUrl() throws Exception {
    when(hostNameResolver.getRotatedAddresses(RESOLVED_HOST_NAME)).thenReturn(asList(notSANAddress));

    Exception e = flowRunner("outerNotSANHost").runExpectingException();
    assertThat(e.getCause(), hasCause(instanceOf(SSLHandshakeException.class)));
    verify(hostNameResolver).getRotatedAddresses(RESOLVED_HOST_NAME);
  }

}
