/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.config.ast;

import static java.lang.Thread.currentThread;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;
import static org.mule.runtime.config.api.dsl.CoreDslConstants.FLOW_IDENTIFIER;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;
import static org.mule.runtime.dsl.api.xml.parser.XmlConfigurationDocumentLoader.noValidationDocumentLoader;
import static org.mule.test.allure.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.test.allure.AllureConstants.ArtifactAst.ParameterAst.PARAMETER_AST;

import org.mule.extension.ws.internal.WebServiceConsumer;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.runtime.config.api.dsl.model.ComponentBuildingDefinitionRegistry;
import org.mule.runtime.config.api.dsl.processor.ArtifactConfig;
import org.mule.runtime.config.internal.model.ApplicationModel;
import org.mule.runtime.core.api.extension.RuntimeExtensionModelProvider;
import org.mule.runtime.core.api.registry.ServiceRegistry;
import org.mule.runtime.core.api.registry.SpiServiceRegistry;
import org.mule.runtime.dsl.api.component.ComponentBuildingDefinitionProvider;
import org.mule.runtime.dsl.api.xml.XmlNamespaceInfoProvider;
import org.mule.runtime.dsl.api.xml.parser.ConfigFile;
import org.mule.runtime.dsl.api.xml.parser.ConfigLine;
import org.mule.runtime.dsl.internal.xml.parser.XmlApplicationParser;
import org.mule.runtime.module.extension.api.loader.java.DefaultJavaExtensionModelLoader;
import org.mule.runtime.module.extension.internal.config.ExtensionBuildingDefinitionProvider;
import org.mule.runtime.module.extension.internal.manager.DefaultExtensionManager;
import org.mule.tck.junit4.AbstractMuleContextTestCase;
import org.mule.test.runner.infrastructure.ExtensionsTestInfrastructureDiscoverer;

import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import javax.xml.parsers.SAXParserFactory;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Test;
import org.w3c.dom.Document;
import org.xml.sax.helpers.DefaultHandler;

@Feature(ARTIFACT_AST)
@Story(PARAMETER_AST)
public class ParameterAstTestCase extends AbstractMuleContextTestCase {

  private ArtifactAst artifactAst;

  @Before
  public void before() throws Exception {
    ArtifactConfig.Builder artifactConfigBuilder = new ArtifactConfig.Builder();

    URL resource = this.getClass().getClassLoader().getResource("ast/parameters-test-config.xml");

    Optional<ConfigLine> configLine;
    ServiceRegistry serviceRegistry = new SpiServiceRegistry();
    try (InputStream configFileStream = resource.openStream()) {
      Document document =
          noValidationDocumentLoader().loadDocument(SAXParserFactory::newInstance, "config", configFileStream,
                                                    new DefaultHandler());

      ImmutableList.Builder<XmlNamespaceInfoProvider> namespaceInfoProvidersBuilder = ImmutableList.builder();
      namespaceInfoProvidersBuilder
          .addAll(serviceRegistry.lookupProviders(XmlNamespaceInfoProvider.class, currentThread().getContextClassLoader()));
      ImmutableList<XmlNamespaceInfoProvider> xmlNamespaceInfoProviders = namespaceInfoProvidersBuilder.build();

      XmlApplicationParser xmlApplicationParser = new XmlApplicationParser(xmlNamespaceInfoProviders);
      configLine = xmlApplicationParser.parse(document.getDocumentElement());
    } catch (IOException e) {
      throw new UncheckedIOException(e);
    }

    artifactConfigBuilder.addConfigFile(new ConfigFile(resource.getFile(), Collections.singletonList(
                                                                                                     configLine
                                                                                                         .orElseThrow(() -> new IllegalArgumentException(String
                                                                                                             .format("Failed to parse %s.",
                                                                                                                     resource))))));

    ArtifactConfig artifactConfig = artifactConfigBuilder.build();

    List<ExtensionModel> runtimeExtensionModels = new ArrayList<>();
    Collection<RuntimeExtensionModelProvider> runtimeExtensionModelProviders = new SpiServiceRegistry()
        .lookupProviders(RuntimeExtensionModelProvider.class, Thread.currentThread().getContextClassLoader());
    for (RuntimeExtensionModelProvider runtimeExtensionModelProvider : runtimeExtensionModelProviders) {
      runtimeExtensionModels.add(runtimeExtensionModelProvider.createExtensionModel());
    }

    DefaultExtensionManager extensionManager = new DefaultExtensionManager();
    muleContext.setExtensionManager(extensionManager);
    initialiseIfNeeded(extensionManager, muleContext);

    ExtensionsTestInfrastructureDiscoverer discoverer = new ExtensionsTestInfrastructureDiscoverer(extensionManager);

    DefaultJavaExtensionModelLoader extensionModelLoader = new DefaultJavaExtensionModelLoader();
    for (Class<?> annotatedClass : new Class[] {WebServiceConsumer.class}) {
      discoverer.discoverExtension(annotatedClass, extensionModelLoader);
    }

    ImmutableSet<ExtensionModel> extensionModels = ImmutableSet.<ExtensionModel>builder()
        .addAll(muleContext.getExtensionManager().getExtensions())
        .addAll(runtimeExtensionModels)
        .build();

    final ComponentBuildingDefinitionRegistry componentBuildingDefinitionRegistry =
        new ComponentBuildingDefinitionRegistry();
    serviceRegistry
        .lookupProviders(ComponentBuildingDefinitionProvider.class, ComponentBuildingDefinitionProvider.class.getClassLoader())
        .forEach(componentBuildingDefinitionProvider -> {
          if (componentBuildingDefinitionProvider instanceof ExtensionBuildingDefinitionProvider) {
            // Ignore extensions building definition provider, we have to test this works fine without parsers
          }
          componentBuildingDefinitionProvider.init();
          componentBuildingDefinitionProvider.getComponentBuildingDefinitions()
              .forEach(componentBuildingDefinitionRegistry::register);
        });

    this.artifactAst = new ApplicationModel(artifactConfig, null, extensionModels, Collections.emptyMap(),
                                            Optional.empty(), of(componentBuildingDefinitionRegistry),
                                            uri -> muleContext.getExecutionClassLoader().getResourceAsStream(uri),
                                            false, getFeatureFlaggingService());
  }

  @Test
  public void groupInlineParametersShowInDslTransform() {
    Optional<ComponentAst> optionalGroupInlineParametersShowInDsl = artifactAst.topLevelComponentsStream()
        .filter(componentAst -> componentAst.getIdentifier().equals(FLOW_IDENTIFIER) &&
            "groupInlineParametersShowInDslTransform".equals(componentAst.getComponentId().orElse(null)))
        .findFirst();
    assertThat(optionalGroupInlineParametersShowInDsl, not(empty()));

    ComponentAst groupInlineParametersShowInDsl = optionalGroupInlineParametersShowInDsl.get();
    ComponentAst eeTransformComponent = groupInlineParametersShowInDsl.directChildrenStream().findFirst()
        .orElseThrow(() -> new AssertionError("Missing ee:transform component"));

    ComponentParameterAst setPayloadComponentParameterAst = eeTransformComponent.getParameter("setPayload");
    assertThat(setPayloadComponentParameterAst, not(nullValue()));
    assertThat(setPayloadComponentParameterAst.getValue().getLeft(), is(nullValue()));
    assertThat(setPayloadComponentParameterAst.getValue().getRight(), not(nullValue()));

    ComponentAst setPayloadComponentAst = (ComponentAst) setPayloadComponentParameterAst.getValue().getRight();
    assertThat(setPayloadComponentAst, not(nullValue()));
    assertThat((String) setPayloadComponentAst.getParameter("script").getValue().getRight(),
               containsString("key: payload.mulesoft.key"));

    ComponentParameterAst variablesComponentParameterAst = eeTransformComponent.getParameter("variables");
    assertThat(variablesComponentParameterAst, not(nullValue()));
    assertThat(variablesComponentParameterAst.getValue().getLeft(), is(nullValue()));
    assertThat(variablesComponentParameterAst.getValue().getRight(), not(nullValue()));

    List<ComponentAst> variablesComponentAst = (List<ComponentAst>) variablesComponentParameterAst.getValue().getRight();
    assertThat(variablesComponentAst, not(nullValue()));
    assertThat(variablesComponentAst, hasSize(1));
    ComponentAst variableComponentAst = variablesComponentAst.get(0);
    assertThat(variableComponentAst.getParameter("variableName").getValue().getRight(), equalTo("httpStatus"));
    assertThat(variableComponentAst.getParameter("script").getValue().getRight(), equalTo("404"));
  }

  @Test
  public void groupInlineParametersShowInDslWsc() {
    Optional<ComponentAst> optionalGroupInlineParametersShowInDsl = artifactAst.topLevelComponentsStream()
        .filter(componentAst -> componentAst.getIdentifier().equals(FLOW_IDENTIFIER) &&
            "groupInlineParametersShowInDslWsc".equals(componentAst.getComponentId().orElse(null)))
        .findFirst();
    assertThat(optionalGroupInlineParametersShowInDsl, not(empty()));

    ComponentAst groupInlineParametersShowInDsl = optionalGroupInlineParametersShowInDsl.get();
    ComponentAst eeTransformComponent = groupInlineParametersShowInDsl.directChildrenStream().findFirst()
        .orElseThrow(() -> new AssertionError("Missing wsc:consume component"));

    ComponentParameterAst bodyComponentParameterAst = eeTransformComponent.getParameter("body");
    assertThat(bodyComponentParameterAst, not(nullValue()));
    assertThat(bodyComponentParameterAst.getValue().getRight(), is(nullValue()));
    assertThat(bodyComponentParameterAst.getValue().getLeft(), equalTo("#[payload]"));

    ComponentParameterAst headersComponentParameterAst = eeTransformComponent.getParameter("headers");
    assertThat(headersComponentParameterAst, not(nullValue()));
    assertThat(headersComponentParameterAst.getValue().getRight(), is(nullValue()));
    assertThat(headersComponentParameterAst.getValue().getLeft(), equalTo("#[vars.headers]"));

    ComponentParameterAst attachmentsComponentParameterAst = eeTransformComponent.getParameter("attachments");
    assertThat(attachmentsComponentParameterAst, not(nullValue()));
    assertThat(attachmentsComponentParameterAst.getValue().getRight(), is(nullValue()));
    assertThat(attachmentsComponentParameterAst.getValue().getLeft(), equalTo("#[vars.attachments]"));
  }

  @Test
  public void localErrorhandler() {
    Optional<ComponentAst> optionalGroupInlineParametersShowInDsl =
        artifactAst.topLevelComponentsStream().filter(componentAst -> componentAst.getIdentifier().equals(FLOW_IDENTIFIER)
            && "errorHandlers".equals(componentAst.getComponentId().orElse(null))).findFirst();
    assertThat(optionalGroupInlineParametersShowInDsl, not(empty()));

    ComponentAst groupInlineParametersShowInDsl = optionalGroupInlineParametersShowInDsl.get();

    List<ComponentAst> children = groupInlineParametersShowInDsl.directChildrenStream().collect(toList());
    assertThat(children, hasSize(2));

    ComponentAst loggerComponent = children.get(0);
    assertThat(loggerComponent.getIdentifier().getName(), is("logger"));

    ComponentAst errorHandler = children.get(1);
    assertThat(errorHandler.getIdentifier().getName(), is("error-handler"));

    ComponentAst onErrorContinueComponent = errorHandler.directChildrenStream().findFirst()
        .orElseThrow(() -> new AssertionError("Missing on-error-continue component"));
    assertThat(onErrorContinueComponent.getIdentifier().getName(), is("on-error-continue"));

    ComponentParameterAst typeParameterAst = onErrorContinueComponent.getParameter("type");
    assertThat(typeParameterAst, not(nullValue()));
    assertThat(typeParameterAst.getValue().getRight(), equalTo("APP:SOME"));
    assertThat(typeParameterAst.getValue().getLeft(), is(nullValue()));
  }

}
