/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.module.service.scheduler;

import static java.lang.Thread.currentThread;
import static java.util.concurrent.TimeUnit.SECONDS;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import static org.mule.test.allure.AllureConstants.SchedulerServiceFeature.SCHEDULER_SERVICE;

import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.scheduler.SchedulerBusyException;
import org.mule.runtime.api.scheduler.SchedulerPoolsConfig;
import org.mule.runtime.api.scheduler.SchedulerPoolsConfigFactory;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.api.util.concurrent.Latch;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import javax.inject.Inject;

import io.qameta.allure.Feature;

@Feature(SCHEDULER_SERVICE)
public class SchedulerConfigurationTestCase extends MuleArtifactFunctionalTestCase {

  @Rule
  public ExpectedException expected = ExpectedException.none();

  @Inject
  private SchedulerPoolsConfig appSchedulerPoolsConfig;

  @Inject
  private SchedulerService schedulerService;

  @Override
  protected String getConfigFile() {
    return "service/scheduler/scheduler-config-app.xml";
  }

  @Test
  public void configInContext() {
    SchedulerPoolsConfigFactory appSchedulerPoolsConfig = (SchedulerPoolsConfigFactory) this.appSchedulerPoolsConfig;
    assertThat(appSchedulerPoolsConfig.getConfig().isPresent(), is(true));

    assertThat(appSchedulerPoolsConfig.getConfig().get().getCpuLightPoolSize().getAsInt(), is(2));
    assertThat(appSchedulerPoolsConfig.getConfig().get().getIoCorePoolSize().getAsInt(), is(1));
    // Default config is a large value (256), so by forcing a low value here we can test based on the difference
    assertThat(appSchedulerPoolsConfig.getConfig().get().getIoMaxPoolSize().getAsInt(), is(1));
    assertThat(appSchedulerPoolsConfig.getConfig().get().getIoQueueSize().getAsInt(), is(0));
    assertThat(appSchedulerPoolsConfig.getConfig().get().getCpuIntensivePoolSize().getAsInt(), is(2));
  }

  @Test
  public void appUsesCustomPools() throws InterruptedException, ExecutionException {
    Latch latch = new Latch();
    final Scheduler schedulerOuter = schedulerService.cpuLightScheduler();
    final Scheduler schedulerInner = schedulerService.ioScheduler();

    executeWork(latch, schedulerOuter, schedulerInner);

    expected.expectCause(instanceOf(SchedulerBusyException.class));

    // since IO pool max size is just one, we expect the second submission to be rejected
    try {
      executeWork(latch, schedulerOuter, schedulerInner);
    } finally {
      latch.countDown();
      schedulerInner.shutdownNow();
      schedulerOuter.shutdownNow();
    }
  }

  private void executeWork(Latch latch, final Scheduler schedulerOuter, final Scheduler schedulerInner)
      throws InterruptedException, ExecutionException {
    final Future<?> submit = schedulerOuter.submit(() -> {
      schedulerInner.submit(() -> {
        try {
          latch.await(DEFAULT_TEST_TIMEOUT_SECS, SECONDS);
        } catch (InterruptedException e) {
          currentThread().interrupt();
        }
      });
      return null;
    });

    submit.get();
  }
}
