/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.module.streaming;

import static java.lang.System.gc;
import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertThat;
import static org.mule.runtime.core.api.util.IOUtils.getResourceAsUrl;
import static org.mule.tck.probe.PollingProber.probe;
import static org.mule.test.allure.AllureConstants.StreamingFeature.STREAMING;
import static org.mule.test.allure.AllureConstants.StreamingFeature.StreamingStory.STREAM_MANAGEMENT;

import com.google.common.collect.ImmutableList;
import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;
import org.junit.Rule;
import org.junit.Test;
import org.mule.functional.api.component.TestConnectorQueueHandler;
import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.core.api.processor.Processor;
import org.mule.runtime.core.api.streaming.StreamingManager;
import org.mule.runtime.core.api.streaming.StreamingStatistics;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.junit4.rule.SystemProperty;
import org.mule.tck.probe.JUnitLambdaProbe;
import org.mule.tck.probe.PollingProber;

import javax.inject.Inject;
import java.lang.ref.PhantomReference;
import java.lang.ref.ReferenceQueue;
import java.util.concurrent.atomic.AtomicInteger;

@Feature(STREAMING)
@Story(STREAM_MANAGEMENT)
public class ClosedCursorProviderTestCase extends MuleArtifactFunctionalTestCase {

  private static final int GC_TIMEOUT_MILLIS = 6000;
  private static final int TIMEOUT_MILLIS = 10000;
  private static final int POLL_DELAY_MILLIS = 100;

  private static AtomicInteger payloadsProcessed;

  @Inject
  protected StreamingManager streamingManager;

  @Rule
  public SystemProperty workingDir =
      new SystemProperty("workingDir", getResourceAsUrl("streaming/", ClosedCursorProviderTestCase.class).getPath());

  @Rule
  public DynamicPort port = new DynamicPort("http.port");

  @Override
  protected String getConfigFile() {
    return "streaming/closed-cursor-provider-config.xml";
  }

  @Override
  protected void doSetUp() throws Exception {
    super.doSetUp();
    payloadsProcessed = new AtomicInteger(0);
  }

  @Test
  @Issue("MULE-19068")
  public void targetReturnManageCursorStreamProvider() throws Exception {
    flowRunner("target-return-manage-cursor-stream-provider").keepStreamsOpen().run();
    TestConnectorQueueHandler queueHandler = new TestConnectorQueueHandler(registry);
    final Message asyncResult = queueHandler.read("async-out", 10 * 1000).getMessage();
    assertNotNull(asyncResult);
    assertEquals("\"access_token\"", getPayloadAsString(asyncResult));
  }

  @Test
  @Issue("MULE-19015")
  public void foreachBatchSizeWithMessageCollectionClosedCursorProvider() throws Exception {
    ImmutableList<Message> payload = ImmutableList.<Message>builder()
        .add(Message.builder()
            .payload(createCursorProviderTypeValue("%dw 2.0\noutput application/json\n---\n {\"id\": \"0\", \"name\": \"Fox\", \"toys\": [\"Frisbee\", \"Ball\", \"Ropes\"]}"))
            .build())
        .add(Message.builder()
            .payload(createCursorProviderTypeValue("%dw 2.0\noutput application/json\n---\n {\"id\": \"0\", \"name\": \"Ipa\", \"toys\": [\"Frisbee\", \"Ball\", \"Ropes\"]}"))
            .build())
        .add(Message.builder()
            .payload(createCursorProviderTypeValue("%dw 2.0\noutput application/json\n---\n {\"id\": \"0\", \"name\": \"Oli\", \"toys\": [\"Frisbee\", \"Ball\", \"Ropes\"]}"))
            .build())
        .add(Message.builder()
            .payload(createCursorProviderTypeValue("%dw 2.0\noutput application/json\n---\n {\"id\": \"0\", \"name\": \"Fox-2\", \"toys\": [\"Frisbee\", \"Ball\", \"Ropes\"]}"))
            .build())
        .add(Message.builder()
            .payload(createCursorProviderTypeValue("%dw 2.0\noutput application/json\n---\n {\"id\": \"0\", \"name\": \"Ipa-2\", \"toys\": [\"Frisbee\", \"Ball\", \"Ropes\"]}"))
            .build())
        .add(Message.builder()
            .payload(createCursorProviderTypeValue("%dw 2.0\noutput application/json\n---\n {\"id\": \"0\", \"name\": \"Oli-2\", \"toys\": [\"Frisbee\", \"Ball\", \"Ropes\"]}"))
            .build())
        .add(Message.builder()
            .payload(createCursorProviderTypeValue("%dw 2.0\noutput application/json\n---\n {\"id\": \"0\", \"name\": \"Fox-3\", \"toys\": [\"Frisbee\", \"Ball\", \"Ropes\"]}"))
            .build())
        .add(Message.builder()
            .payload(createCursorProviderTypeValue("%dw 2.0\noutput application/json\n---\n {\"id\": \"0\", \"name\": \"Ipa-3\", \"toys\": [\"Frisbee\", \"Ball\", \"Ropes\"]}"))
            .build())
        .add(Message.builder()
            .payload(createCursorProviderTypeValue("%dw 2.0\noutput application/json\n---\n {\"id\": \"0\", \"name\": \"Oli-3\", \"toys\": [\"Frisbee\", \"Ball\", \"Ropes\"]}"))
            .build())
        .build();
    flowRunner("foreach-batch-size-message-collection-closed-cursor-provider-flow").withPayload(payload)
        .run();
    assertThat(payloadsProcessed.get(), is(3));
    assertAllStreamingResourcesClosed();
  }

  private TypedValue createCursorProviderTypeValue(String expression) {
    return muleContext.getExpressionManager().evaluate(expression);
  }

  @Test
  @Issue("MULE-19015")
  public void foreachBatchSizeClosedCursorProvider() throws Exception {
    flowRunner("foreach-batch-size-closed-cursor-provider-flow").run();
    assertThat(payloadsProcessed.get(), is(3));

    assertAllStreamingResourcesClosed();
  }

  @Test
  @Issue("MULE-18573")
  public void foreachClosedCursorProvider() throws Exception {
    flowRunner("foreach-closed-cursor-provider-flow").run();
    assertThat(payloadsProcessed.get(), is(27));

    assertAllStreamingResourcesClosed();
  }

  @Test
  @Issue("MULE-18573")
  public void parallelForeachClosedCursorProvider() throws Exception {
    flowRunner("parallel-foreach-closed-cursor-provider-flow").run();
    assertThat(payloadsProcessed.get(), is(27));

    assertAllStreamingResourcesClosed();
  }

  private void assertAllStreamingResourcesClosed() {
    StreamingStatistics stats = streamingManager.getStreamingStatistics();
    new PollingProber(TIMEOUT_MILLIS, POLL_DELAY_MILLIS).check(new JUnitLambdaProbe(() -> {
      assertThat("There're still open cursor providers", stats.getOpenCursorProvidersCount(), is(0));
      assertThat("There're still open cursors", stats.getOpenCursorsCount(), is(0));
      return true;
    }));
  }

  public static class GarbageCollectorRunner implements Processor {

    @Override
    public CoreEvent process(CoreEvent event) throws MuleException {
      // Ensure GC run
      Object object = new Object();
      PhantomReference<Object> objectRef = new PhantomReference<>(object, new ReferenceQueue<>());
      object = null;
      probe(GC_TIMEOUT_MILLIS, POLL_DELAY_MILLIS, () -> {
        gc();
        assertThat(objectRef.isEnqueued(), is(true));
        return true;
      });

      return event;
    }
  }

  public static class ForeachCounterCaptor implements Processor {

    @Override
    public CoreEvent process(CoreEvent event) throws MuleException {
      payloadsProcessed.incrementAndGet();

      return event;
    }
  }
}
