/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.module.streaming;

import static com.mulesoft.mule.runtime.core.internal.config.EEMuleProperties.KRYO_OBJECT_SERIALIZER;
import static org.apache.commons.lang3.RandomStringUtils.randomAlphabetic;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.junit.Assert.assertThat;
import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.api.serialization.ObjectSerializer;
import org.mule.runtime.api.streaming.bytes.CursorStream;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.core.api.util.IOUtils;

import javax.inject.Inject;
import javax.inject.Named;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Test;

@Feature("Streaming")
@Story("Object Streaming")
public class EEByteStreamingExtensionTestCase extends MuleArtifactFunctionalTestCase {

  @Inject
  @Named(KRYO_OBJECT_SERIALIZER)
  private ObjectSerializer objectSerializer;

  private String data = randomAlphabetic(2048);

  @Override
  protected String getConfigFile() {
    return "streaming/ee-byte-streaming-extension-config.xml";
  }

  @Test
  @Description("Consume a stream using a file store")
  public void fileStoreStreaming() throws Exception {
    CursorStreamProvider provider = getStreamProvider();
    try (CursorStream cursor = provider.openCursor()) {
      assertThat(IOUtils.toString(cursor), equalTo(data));
      final int rewindPosition = data.length() - 10;
      cursor.seek(rewindPosition);
      assertThat(IOUtils.toString(cursor), equalTo(data.substring(rewindPosition)));
    }
  }

  @Test
  @Description("Serialize a cursor using kryo")
  public void serializeWithKryo() throws Exception {
    CursorStreamProvider provider = getStreamProvider();

    assertThat(objectSerializer.getInternalProtocol().getClass().getName(),
               equalTo("com.mulesoft.mule.runtime.module.serialization.kryo.internal.protocol.KryoSerializationProtocol"));
    byte[] bytes = objectSerializer.getInternalProtocol().serialize(provider);
    CursorStreamProvider deserializedProvider = objectSerializer.getInternalProtocol().deserialize(bytes);
    assertThat(IOUtils.toString(deserializedProvider.openCursor()), equalTo(data));
  }

  private CursorStreamProvider getStreamProvider() throws Exception {
    return (CursorStreamProvider) flowRunner("toStream").keepStreamsOpen()
        .withPayload(data)
        .run().getMessage().getPayload().getValue();
  }
}
