/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.module.streaming;

import static org.apache.commons.lang3.RandomStringUtils.randomAlphabetic;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.core.IsNull.notNullValue;
import static org.junit.Assert.assertThat;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.junit.Before;
import org.junit.Test;
import org.mule.functional.api.component.TestConnectorQueueHandler;
import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.api.streaming.object.CursorIterator;
import org.mule.runtime.api.streaming.object.CursorIteratorProvider;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature("Streaming")
@Story("Object Streaming")
public class EEObjectStreamingExtensionTestCase extends MuleArtifactFunctionalTestCase {

  private List<String> data = new ArrayList<>();

  private TestConnectorQueueHandler queueHandler;

  @Override
  protected String getConfigFile() {
    return "streaming/ee-object-streaming-extension-config.xml";
  }

  @Before
  public void before() throws IOException {
    queueHandler = new TestConnectorQueueHandler(registry);
  }

  @Override
  protected void doSetUp() throws Exception {
    super.doSetUp();
    final int dataSize = 1000;

    data = new ArrayList<>(dataSize);
    for (int i = 0; i < dataSize; i++) {
      data.add(randomAlphabetic(10));
    }
    super.doSetUp();
  }

  @Test
  @Description("Consume a stream using a file store")
  public void fileStoreStreaming() throws Exception {
    testWith("createAndConsumeFileStoreStream");
  }

  @Test
  @Description("Consume an iterator using a file store buffer and forcing an bucket overflow")
  @Issue("EE-7610")
  public void fileStoreStreamingWithBucketOverflow() throws Exception {
    testWith("createAndConsumeFileStoreStreamWithBucketOverflow");
    // assert that the payload was also correctly received (deserialized) through the vm queue
    assertThat(queueHandler.read("processed", RECEIVE_TIMEOUT), notNullValue());
  }

  private void testWith(String flow) throws Exception, IOException {
    CursorIteratorProvider provider =
        (CursorIteratorProvider) flowRunner(flow).keepStreamsOpen()
            .withPayload(data)
            .run().getMessage().getPayload().getValue();


    try (CursorIterator cursor = provider.openCursor()) {
      assertThat(cursor.getSize(), is(data.size()));
      for (int i = 0; cursor.hasNext(); i++) {
        assertThat(cursor.next(), is(data.get(i)));
      }

      cursor.seek(cursor.getSize() - 1);
      assertThat(cursor.next(), is(data.get(data.size() - 1)));
    }
  }
}
