/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.transaction;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.collection.IsIterableContainingInAnyOrder.containsInAnyOrder;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.hamcrest.core.Every.everyItem;
import static org.hamcrest.core.IsNot.not;
import static org.mule.test.allure.AllureConstants.ComponentsFeature.FlowReferenceStory.FLOW_REFERENCE;
import static org.mule.test.allure.AllureConstants.RoutersFeature.AsyncStory.ASYNC;
import static org.mule.test.allure.AllureConstants.RoutersFeature.ParallelForEachStory.PARALLEL_FOR_EACH;
import static org.mule.test.allure.AllureConstants.RoutersFeature.ScatterGatherStory.SCATTER_GATHER;
import static org.mule.test.allure.AllureConstants.RoutersFeature.UntilSuccessfulStory.UNTIL_SUCCESSFUL;
import static org.mule.test.allure.AllureConstants.TransactionFeature.TRANSACTION;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Rule;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.test.allure.AllureConstants;
import org.mule.test.runner.RunnerDelegateTo;

import org.junit.Test;
import org.junit.runners.Parameterized;

import io.qameta.allure.Description;
import io.qameta.allure.Issue;

@RunnerDelegateTo(Parameterized.class)
@Story(TRANSACTION)
@Story(PARALLEL_FOR_EACH)
@Story(SCATTER_GATHER)
@Story(UNTIL_SUCCESSFUL)
@Story(ASYNC)
@Story(FLOW_REFERENCE)
@Feature(AllureConstants.ErrorHandlingFeature.ERROR_HANDLING)
public class TransactionsWithRoutersTestCase extends AbstractTransactionWithRoutersTestCase {

  @Rule
  public DynamicPort httpPort = new DynamicPort("http.port");

  public TransactionsWithRoutersTestCase(String processingStrategyFactoryClassname) {
    super(processingStrategyFactoryClassname);
  }

  @Override
  protected String getTransactionConfigFile() {
    return "transaction/transaction-routers.xml";
  }

  @Test
  @Description("When running inside a tx, parallel foreach should work as common foreach")
  public void parallelForeachInSameThread() throws Exception {
    runsInSameTransaction("txParallelForeach", TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  public void parallelForeachInsideParallelForeachInSameThread() throws Exception {
    runsInSameTransaction("txParallelForeachInsideParallelForeach", TX_MESSAGE, OTHER_TX_MESSAGE, OTHER_TX_MESSAGE, TX_MESSAGE);
  }

  @Test
  @Description("Error handling of parallel foreach does not change even in context of transactions, where execution is sequential")
  public void parallelForEachHasSameErrorHandling() throws Exception {
    runsInSameTransactionWithErrors("txParallelForeachWithErrors");
  }

  @Test
  @Description("When running inside a tx, every route executes sequentially")
  public void scatterGatherRunsInSameThread() throws Exception {
    runsInSameTransaction("txScatterGather", TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  @Description("Error handling of scatter gather does not change even in context of transactions, where execution is sequential")
  public void scatterGatherHasSameErrorHandling() throws Exception {
    runsInSameTransactionWithErrors("txScatterGatherWithErrors");
  }

  @Test
  @Description("When running inside a tx, every execution of until successful must be in the same thread")
  public void untilSucessfulRunsInSameThread() throws Exception {
    runsInSameTransaction("txUntilSuccessful", TX_MESSAGE, TX_MESSAGE, TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  @Description("When running inside a tx, every execution of until successful must be in the same thread")
  public void untilSucessfulWithErrorHandlerWithRouterRunsInSameThread() throws Exception {
    runsInSameTransaction("txUntilSuccessfulOtherError", TX_MESSAGE, TX_MESSAGE, TX_MESSAGE, OTHER_TX_MESSAGE, TX_MESSAGE,
                          TX_MESSAGE,
                          TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  @Description("When running inside a tx, async runs in another thread and it's not part of the tx")
  public void asyncNotInTx() throws Exception {
    flowRunner("txAsync").run();
    latch.await();
    assertThat(runsInTx, containsInAnyOrder(true, false));
    assertThat(payloads, hasSize(1));
    assertThat(payloads.get(0), is(TX_MESSAGE));
  }

  @Test
  @Description("When running in non-tx and use flow-ref, even if the source of such flow begins a tx, the flow runs without tx")
  public void flowRefWithoutTx() throws Exception {
    flowRunner("flowRefWithoutTx").run();
    assertThat(threads, hasSize(2));
    assertThat(runsInTx, everyItem(is(false)));
    assertThat(payloads, contains(TX_MESSAGE, OTHER_TX_MESSAGE));
  }

  @Test
  @Description("When running inside tx and use flow-ref, the tx is propagated")
  public void flowRefWithTx() throws Exception {
    runsInSameTransaction("flowRefWithTx", TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  @Description("When running inside tx and use flow-ref, the tx is propagated even in case of subflow")
  public void flowRefWithTxToSubFlow() throws Exception {
    runsInSameTransaction("flowRefToSubFlowWithTx", TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  @Description("When running inside tx and use flow-ref, the tx is propagated even in case of dynamic subflow")
  public void flowRefWithTxToDynamicSubFlow() throws Exception {
    runsInSameTransaction("flowRefToDynamicSubFlowWithTx", TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  public void flowRefWithTxToFlowWithError() throws Exception {
    runsInSameTransaction("flowRefToFlowWithErrorTx", TX_MESSAGE, TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  public void flowRefDynamicWithTxToFlowWithError() throws Exception {
    runsInSameTransaction("flowRefDynamicToFlowWithErrorTx", TX_MESSAGE, TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  @Description("When Flow that creates tx has flow-ref to flow that raises error and handles it with on-error-continue, then tx must go on in the first flow")
  public void flowRefToFlowWithErrorAndOnErrorContinue() throws Exception {
    runsInSameTransaction("flowRefToFlowWithErrorAndOnErrorContinue", TX_MESSAGE, TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  public void flowRefDynamicToFlowWithTxAndErrorWithOnErrorPropagate() throws Exception {
    flowRunner("flowRefToTxFlowWithError").withVariable("errorType", "raise-propagate-error").run();
    assertThat(threads, hasSize(4));
    // when executing the on-error-propagate, tx has already been rolled back
    assertThat(runsInTx, contains(false, true, false, false));
    assertThat(payloads, contains(TX_MESSAGE, OTHER_TX_MESSAGE, OTHER_TX_MESSAGE, TX_MESSAGE));
  }

  @Test
  public void flowRefDynamicToFlowWithTxAndErrorWithOnErrorContinue() throws Exception {
    flowRunner("flowRefToTxFlowWithError").withVariable("errorType", "raise-continue-error").run();
    assertThat(threads, hasSize(3));
    // when executing the on-error-continue, tx has not yet been committed (it still runs as part of the tx)
    assertThat(runsInTx, contains(false, true, true));
    assertThat(payloads, contains(TX_MESSAGE, OTHER_TX_MESSAGE, OTHER_TX_MESSAGE));
  }

  @Test
  public void flowRefWithTxToFlowWithErrorAndFlowRefInErrorHandler() throws Exception {
    runsInSameTransaction("flowRefToFlowWithErrorTxAndFlowRef", TX_MESSAGE, TX_MESSAGE, OTHER_TX_MESSAGE, TX_MESSAGE);
  }

  @Test
  public void nestedTries() throws Exception {
    runsInSameTransaction("nestedTries", TX_MESSAGE, TX_MESSAGE, TX_MESSAGE);
  }

  @Test
  public void nestedTriesContinuesTx() throws Exception {
    runsInSameTransaction("nestedTriesContinuesTx", TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  public void flowWithTxSourceWithTryContinuesTx() throws Exception {
    runsInSameTransactionAsync("toQueueFlowWithTxSourceWithTryContinuesTx", TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  public void nestedTriesWithIndifferentInTheMiddle() throws Exception {
    runsInSameTransaction("nestedTriesWithIndifferentInTheMiddle", TX_MESSAGE, TX_MESSAGE, TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  public void flowWithTxSourceAndFlowRef() throws Exception {
    runsInSameTransactionAsync("toQueue", OTHER_TX_MESSAGE, TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  public void flowRefToFlowWithErrorAndContinue() throws Exception {
    runsInSameTransaction("flowRefToFlowWithErrorAndContinue", TX_MESSAGE, TX_MESSAGE, OTHER_TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  public void flowRefToFlowWithErrorAndPropagate() throws Exception {
    // Since the on-error-propagate is not in the flow/try that created the tx, it should not rollback it. Thus, it should
    // still run in the same thread (and within a tx)
    runsInSameTransaction("flowRefToFlowWithErrorAndPropagate", TX_MESSAGE, TX_MESSAGE, OTHER_TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  public void nestedTriesWithOnErrorPropagates() throws Exception {
    runsInSameTransaction("nestedTriesWithOnErrorPropagate", TX_MESSAGE, TX_MESSAGE, OTHER_TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  public void innerTryWithOnErrorPropagate() throws Exception {
    runsInSameTransaction("tryWithInnerTryWithOnErrorPropagate", TX_MESSAGE, TX_MESSAGE, OTHER_TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  public void nestedTriesWithOnErrorContinue() throws Exception {
    runsInSameTransaction("nestedTriesWithOnErrorContinue", TX_MESSAGE, TX_MESSAGE, OTHER_TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  public void onErrorPropagateRaisesError() throws Exception {
    runsInSameTransaction("onErrorPropagateRaisesError", TX_MESSAGE, TX_MESSAGE, OTHER_TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  public void onErrorContinueRaisesError() throws Exception {
    runsInSameTransaction("onErrorContinueRaisesError", TX_MESSAGE, TX_MESSAGE, OTHER_TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  public void onErrorContinueAndPropagateRaiseError() throws Exception {
    runsInSameTransaction("onErrorContinueAndPropagateRaiseError", TX_MESSAGE, TX_MESSAGE, OTHER_TX_MESSAGE, OTHER_TX_MESSAGE,
                          OTHER_TX_MESSAGE);
  }

  @Test
  public void nestedTryRollbacksTxInInnerTry() throws Exception {
    flowRunner("nestedTryRollbacksTxInInnerTry").run();
    assertThat(runsInTx, contains(false, true, true, false));
  }

  @Test
  public void tryWithinTryDoesNotFinishTx() throws Exception {
    runsInSameTransaction("tryWithinTryDoesNotFinishTx", TX_MESSAGE, OTHER_TX_MESSAGE, OTHER_TX_MESSAGE);
  }

  @Test
  public void flowRefToFlowWithErrorPropagateWithError() throws Exception {
    runsInSameTransaction("flowRefToFlowWithErrorPropagateWithError", TX_MESSAGE, OTHER_TX_MESSAGE, OTHER_TX_MESSAGE,
                          OTHER_TX_MESSAGE);
  }

  @Test
  public void tryRunsInSameThreadAsBeforeExecuting() throws Exception {
    flowRunner("tryRunsInSameThreadAsBeforeExecuting").run();
    assertThat(threads.get(1), is(threads.get(0)));
    onProcessingStrategy(() -> assertThreadType(threads.get(0), UBER), () -> assertThreadType(threads.get(0), CPU_LIGHT));
  }

  @Test
  public void tryWithAlwaysBegin() throws Exception {
    flowRunner("tryWithAlwaysBegin").run();
    onProcessingStrategy(() -> {
      assertThat(threads.get(1), is(threads.get(0)));
      assertThreadType(threads.get(0), UBER);
    }, () -> {
      assertThat(threads.get(1), not(threads.get(0)));
      assertThreadType(threads.get(0), CPU_LIGHT);
      assertThreadType(threads.get(1), IO);
    });
  }

  @Test
  public void tryWithBeginOrJoin() throws Exception {
    flowRunner("tryWithBeginOrJoin").run();
    onProcessingStrategy(() -> {
      assertThat(threads.get(1), is(threads.get(0)));
      assertThreadType(threads.get(0), UBER);
    }, () -> {
      assertThat(threads.get(1), not(threads.get(0)));
      assertThreadType(threads.get(0), CPU_LIGHT);
      assertThreadType(threads.get(1), IO);
    });
  }

  @Test
  public void tryWithBeginOrJoinNestedIndifferent() throws Exception {
    flowRunner("tryWithBeginOrJoinNestedIndifferent").run();
    onProcessingStrategy(() -> {
      assertThat(threads.get(1), is(threads.get(0)));
      assertThreadType(threads.get(0), UBER);
    }, () -> {
      assertThat(threads.get(1), not(threads.get(0)));
      assertThat(threads.get(2), is(threads.get(1)));
      assertThreadType(threads.get(0), CPU_LIGHT);
      assertThreadType(threads.get(1), IO);
    });
  }

  @Test
  public void tryWithBeginOrJoinNestedBeginOrJoin() throws Exception {
    flowRunner("tryWithBeginOrJoinNestedBeginOrJoin").run();
    onProcessingStrategy(() -> {
      assertThat(threads.stream().allMatch(t -> t == threads.get(0)), is(true));
      assertThreadType(threads.get(0), UBER);
    }, () -> {
      assertThat(threads.get(1), not(threads.get(0)));
      assertThat(threads.get(2), is(threads.get(1)));
      assertThreadType(threads.get(0), CPU_LIGHT);
      assertThreadType(threads.get(1), IO);
    });
  }

  @Test
  @Issue("MULE-18488")
  public void txTryWithSdkOperation() throws Exception {
    // execute twice to manifest the issue
    flowRunner("txTryWithSdkOperation").run();
    flowRunner("txTryWithSdkOperation").run();
  }

}
