/*
 * (c) 2003-2023 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.batch;

import static org.mule.runtime.http.api.HttpConstants.Method.GET;

import static java.util.Collections.synchronizedList;
import static java.util.Arrays.asList;

import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;

import org.mule.runtime.api.component.AbstractComponent;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.security.Authentication;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.core.api.processor.Processor;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.TestHttpClient;
import org.mule.tck.junit4.rule.DynamicPort;

import java.util.ArrayList;
import java.util.Base64;
import java.util.List;

import io.qameta.allure.Description;
import io.qameta.allure.Issue;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class BatchEventAuthenticationTestCase extends AbstractBatchTestCase {

  private static List<Object> payloads;
  private static final int HTTP_STATUS_OK = 200;
  public static final int RESPONSE_TIMEOUT = 5000;

  @ClassRule
  public static final TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Rule
  public DynamicPort port = new DynamicPort("port");

  @Rule
  public TestHttpClient httpClient = new TestHttpClient.Builder(getService(HttpService.class)).build();

  public static class Captor extends AbstractComponent implements Processor {

    @Override
    public CoreEvent process(CoreEvent event) throws MuleException {
      Authentication token =
          (Authentication) (event.getSecurityContext()
              .getAuthentication());
      assertThat(token, is(notNullValue()));
      assertThat(token.getPrincipal(), is(notNullValue()));
      payloads.add(event.getMessage().getPayload().getValue());
      return event;
    }
  }

  @Override
  protected void doSetUp() throws Exception {
    super.doSetUp();
    payloads = synchronizedList(new ArrayList<>());
  }

  @Override
  protected String getConfigFile() {
    return "batch/batchAuthentication.xml";
  }

  @Test
  @Issue("W-11493901")
  @Description("verifiy that SecurityContext is passed to Batch steps")
  public void execute() throws Exception {
    List<String> expectedResult = asList("admin", "admin", "admin");
    HttpRequest request = HttpRequest.builder().uri("http://localhost:" + port.getNumber() + "/test").method(GET)
        .addHeader("Authorization", getBasicAuthenticationHeader("admin", "admin")).build();

    HttpResponse response = httpClient.send(request, HttpRequestOptions.builder().responseTimeout(RESPONSE_TIMEOUT).build());
    assertThat(response.getStatusCode(), is(HTTP_STATUS_OK));
    awaitJobTermination();

    assertThat(payloads, is(expectedResult));
  }

  private static final String getBasicAuthenticationHeader(String username, String password) {
    String valueToEncode = username + ":" + password;
    return "Basic " + Base64.getEncoder().encodeToString(valueToEncode.getBytes());
  }
}
