/*
 * (c) 2003-2023 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.component;

import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;
import static org.mule.runtime.api.component.location.Location.builderFromStringRepresentation;
import static org.mule.runtime.config.api.LazyComponentInitializer.LAZY_COMPONENT_INITIALIZER_SERVICE_KEY;
import static org.mule.test.allure.AllureConstants.ConfigurationComponentLocatorFeature.CONFIGURATION_COMPONENT_LOCATOR;
import static org.mule.test.allure.AllureConstants.ConfigurationComponentLocatorFeature.ConfigurationComponentLocatorStory.SEARCH_CONFIGURATION;

import static java.util.stream.Collectors.toList;

import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;

import org.mule.functional.junit4.MuleArtifactFunctionalTestCase;
import org.mule.runtime.api.component.Component;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.config.api.LazyComponentInitializer;
import org.mule.runtime.core.api.construct.Flow;
import org.mule.runtime.core.api.event.CoreEvent;

import java.util.List;

import javax.inject.Inject;
import javax.inject.Named;

import org.junit.Test;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(CONFIGURATION_COMPONENT_LOCATOR)
@Story(SEARCH_CONFIGURATION)
public class LazyInitConfigurationComponentLocatorTestCase extends MuleArtifactFunctionalTestCase {

  private static final int TOTAL_NUMBER_OF_LOCATIONS = 45;

  @Inject
  @Named(value = LAZY_COMPONENT_INITIALIZER_SERVICE_KEY)
  private LazyComponentInitializer lazyComponentInitializer;

  @Override
  protected String[] getConfigFiles() {
    return new String[] {"component/batch-component-locator-config.xml",
        "component/bitronix-component-locator-config.xml",
        "component/websockets-component-locator-config.xml",
        "component/cache-component-locator-config.xml"};
  }

  @Override
  public boolean enableLazyInit() {
    return true;
  }

  @Description("Lazy init should not create components until an operation is done")
  @Test
  public void lazyInitCalculatesLocations() {
    List<String> allLocations = locator
        .findAllLocations()
        .stream()
        .map(ComponentLocation::getLocation)
        .collect(toList());
    assertThat(allLocations.toString(), allLocations,
               containsInAnyOrder("listenerTlsContext",
                                  "listenerTlsContext/0",

                                  "listenerTlsConfig",
                                  "listenerTlsConfig/connection",

                                  "mainFlow",
                                  "mainFlow/source",
                                  "mainFlow/processors/0",

                                  "batchWithTwoStepsFlow/processors/0",
                                  "batchWithTwoStepsFlow/processors/0/route/0",
                                  "batchWithTwoStepsFlow/processors/0/route/0/route/0",
                                  "batchWithTwoStepsFlow/processors/0/route/0/route/0/processors/0",
                                  "batchWithTwoStepsFlow/processors/0/route/0/route/1",
                                  "batchWithTwoStepsFlow/processors/0/route/0/route/1/processors/0",
                                  "batchWithTwoStepsFlow/processors/0/route/1",
                                  "batchWithTwoStepsFlow/processors/0/route/1/processors/0",

                                  "ws",
                                  "ws/connection",
                                  "ws/connection/0",
                                  "ws/connection/1/0/0",
                                  "ws/connection/1/0/1",
                                  "ws/connection/1/1/0",
                                  "ws/connection/1/1/1",

                                  "antiSocialChat/source",
                                  "antiSocialChat/processors/0",

                                  "ws/connection/1",
                                  "httpListenerConfig",
                                  "httpListenerConfig/connection",

                                  "xa-transactionFlow",
                                  "xa-transactionFlow/processors/0",
                                  "xa-transactionFlow/processors/0/processors/0",

                                  "flowWithCache",
                                  "flowWithCache/processors/0",
                                  "flowWithCache/processors/1",
                                  "flowWithCache/processors/2",
                                  "flowWithCache/processors/2/processors/0",

                                  "requestTlsContextWithCertificate",
                                  "requestTlsContextWithCertificate/0",

                                  "requestConfigWithCertificate",
                                  "requestConfigWithCertificate/connection",
                                  "requestConfigWithCertificate/connection/0",
                                  "requestConfigWithCertificate/connection/1",

                                  "batchWithTwoStepsFlow",

                                  "antiSocialChat",

                                  "cachingStrategy",

                                  "null"));
  }

  @Test
  public void cacheComponent() {
    Location location = builderFromStringRepresentation("flowWithCache").build();

    lazyComponentInitializer.initializeComponent(location);
    assertThat(locator.find(location), is(notNullValue()));

    assertThat(locator.find(Location.builderFromStringRepresentation("cachingStrategy").build()), is(notNullValue()));
  }

  @Issue("EE-6201: Cannot lazyInitialize a component twice when using batch")
  @Test
  public void enableProcessorOnBatchStepTwice() {
    Location locationFirstStep =
        builderFromStringRepresentation("batchWithTwoStepsFlow/processors/0/route/0/route/0/processors/0").build();

    lazyComponentInitializer.initializeComponent(locationFirstStep);
    assertThat(locator.findAllLocations(), hasSize(TOTAL_NUMBER_OF_LOCATIONS));
    assertThat(locator.find(locationFirstStep), is(notNullValue()));

    Location locationSecondStep =
        builderFromStringRepresentation("batchWithTwoStepsFlow/processors/0/route/0/route/1/processors/0").build();

    lazyComponentInitializer.initializeComponent(locationSecondStep);
    assertThat(locator.findAllLocations(), hasSize(TOTAL_NUMBER_OF_LOCATIONS));
    assertThat(locator.find(locationSecondStep), is(notNullValue()));
  }

  @Issue("EE-6435")
  @Test
  public void btiTransactionManagerGlobalElement() throws Exception {
    lazyComponentInitializer.initializeComponent(builderFromStringRepresentation("xa-transactionFlow").build());
    assertThat(locator.findAllLocations(), hasSize(TOTAL_NUMBER_OF_LOCATIONS));
    CoreEvent event = flowRunner("xa-transactionFlow").run();
    assertThat(event.getMessage().getPayload().getValue(), equalTo("worked"));
  }

  @Issue("MULE-16600")
  @Test
  public void webSocketsConfig() {
    lazyComponentInitializer.initializeComponent(builderFromStringRepresentation("antiSocialChat").build());
    assertThat(locator.findAllLocations(), hasSize(TOTAL_NUMBER_OF_LOCATIONS));
  }

  @Issue("MULE-18080")
  @Test
  public void webSocketsConfigTlsContextReferenceClientSettings() {
    lazyComponentInitializer.initializeComponent(builderFromStringRepresentation("requestConfigWithCertificate").build());
    assertThat(locator.find(builderFromStringRepresentation("requestTlsContextWithCertificate").build()), is(notNullValue()));
  }

  @Issue("MULE-16600")
  @Test
  public void disposeComponentsFromLocator() {
    lazyComponentInitializer.initializeComponent(builderFromStringRepresentation("antiSocialChat").build());
    assertThat(locator.findAllLocations(), hasSize(TOTAL_NUMBER_OF_LOCATIONS));
    List<Component> flows = locator.find(buildFromStringRepresentation("flow"));
    assertThat(flows, hasSize(1));
    assertThat(((Flow) flows.get(0)).getName(), equalTo("antiSocialChat"));

    // Initialize another flow
    lazyComponentInitializer.initializeComponent(builderFromStringRepresentation("mainFlow").build());
    assertThat(locator.findAllLocations(), hasSize(TOTAL_NUMBER_OF_LOCATIONS));
    flows = locator.find(buildFromStringRepresentation("flow"));
    assertThat(flows, hasSize(1));
    assertThat(((Flow) flows.get(0)).getName(), equalTo("mainFlow"));
  }

}
