/*
 * (c) 2003-2023 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.test.config.ast;

import static org.mule.runtime.api.meta.model.parameter.ParameterGroupModel.DEFAULT_GROUP_NAME;
import static org.mule.runtime.config.api.dsl.CoreDslConstants.FLOW_IDENTIFIER;
import static org.mule.test.allure.AllureConstants.ArtifactAst.ARTIFACT_AST;
import static org.mule.test.allure.AllureConstants.ArtifactAst.ParameterAst.PARAMETER_AST;

import static com.mulesoft.test.allure.AllureConstants.EeComponentsFeature.DynamicEvaluateStory.DYNAMIC_EVALUATE;

import static java.util.Optional.empty;
import static java.util.stream.Collectors.toList;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;
import static org.junit.Assume.assumeThat;

import org.mule.extension.db.internal.DbConnector;
import org.mule.extension.http.internal.temporary.HttpConnector;
import org.mule.extension.socket.api.SocketsExtension;
import org.mule.extension.ws.internal.WebServiceConsumer;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.ast.api.ComponentParameterAst;
import org.mule.test.config.ast.BaseParameterAstTestCase;

import java.util.List;
import java.util.Optional;

import org.junit.Test;

import org.hamcrest.Matchers;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

@Feature(ARTIFACT_AST)
@Story(PARAMETER_AST)
public class ParameterAstEETestCase extends BaseParameterAstTestCase {

  public ParameterAstEETestCase(boolean serialize, boolean populateGenerationInformation) {
    super(serialize, populateGenerationInformation);
  }

  @Test
  public void groupInlineParametersShowInDslTransform() {
    ArtifactAst artifactAst = buildArtifactAst("parameters-test-config.xml",
                                               WebServiceConsumer.class);

    Optional<ComponentAst> optionalGroupInlineParametersShowInDsl = artifactAst.topLevelComponentsStream()
        .filter(componentAst -> componentAst.getIdentifier().equals(FLOW_IDENTIFIER) &&
            "groupInlineParametersShowInDslTransform".equals(componentAst.getComponentId().orElse(null)))
        .findFirst();
    assertThat(optionalGroupInlineParametersShowInDsl, not(empty()));

    ComponentAst groupInlineParametersShowInDsl = optionalGroupInlineParametersShowInDsl.get();
    ComponentAst eeTransformComponent = groupInlineParametersShowInDsl.directChildrenStream().findFirst()
        .orElseThrow(() -> new AssertionError("Missing ee:transform component"));

    ComponentParameterAst setPayloadComponentParameterAst = eeTransformComponent.getParameter("Message", "setPayload");
    assertThat(setPayloadComponentParameterAst, not(nullValue()));
    assertThat(setPayloadComponentParameterAst.getValue().getLeft(), is(nullValue()));
    assertThat(setPayloadComponentParameterAst.getValue().getRight(), not(nullValue()));

    ComponentAst setPayloadComponentAst = (ComponentAst) setPayloadComponentParameterAst.getValue().getRight();
    assertThat(setPayloadComponentAst, not(nullValue()));
    assertThat((String) setPayloadComponentAst.getParameter("SetPayload", "script").getValue().getRight(),
               containsString("key: payload.mulesoft.key"));

    ComponentParameterAst variablesComponentParameterAst = eeTransformComponent.getParameter("Set Variables", "variables");
    assertThat(variablesComponentParameterAst, not(nullValue()));
    assertThat(variablesComponentParameterAst.getValue().getLeft(), is(nullValue()));
    assertThat(variablesComponentParameterAst.getValue().getRight(), not(nullValue()));

    List<ComponentAst> variablesComponentAst = (List<ComponentAst>) variablesComponentParameterAst.getValue().getRight();
    assertThat(variablesComponentAst, not(nullValue()));
    assertThat(variablesComponentAst, hasSize(1));
    ComponentAst variableComponentAst = variablesComponentAst.get(0);
    assertThat(variableComponentAst.getParameter("SetVariable", "variableName").getValue().getRight(),
               equalTo("httpStatus"));
    assertThat(variableComponentAst.getParameter("SetVariable", "script").getValue().getRight(), equalTo("404"));
  }

  @Test
  public void groupInlineParametersShowInDslWsc() {
    ArtifactAst artifactAst = buildArtifactAst("parameters-test-config.xml",
                                               WebServiceConsumer.class);

    Optional<ComponentAst> optionalGroupInlineParametersShowInDsl = artifactAst.topLevelComponentsStream()
        .filter(componentAst -> componentAst.getIdentifier().equals(FLOW_IDENTIFIER) &&
            "groupInlineParametersShowInDslWsc".equals(componentAst.getComponentId().orElse(null)))
        .findFirst();
    assertThat(optionalGroupInlineParametersShowInDsl, not(empty()));

    ComponentAst groupInlineParametersShowInDsl = optionalGroupInlineParametersShowInDsl.get();
    ComponentAst wscConsume = groupInlineParametersShowInDsl.directChildrenStream().findFirst()
        .orElseThrow(() -> new AssertionError("Missing wsc:consume component"));

    ComponentParameterAst bodyComponentParameterAst = wscConsume.getParameter("Message", "body");
    assertThat(bodyComponentParameterAst, not(nullValue()));
    assertThat(bodyComponentParameterAst.getValue().getRight(), is(nullValue()));
    assertThat(bodyComponentParameterAst.getValue().getLeft(), equalTo("payload"));

    ComponentParameterAst headersComponentParameterAst = wscConsume.getParameter("Message", "headers");
    assertThat(headersComponentParameterAst, not(nullValue()));
    assertThat(headersComponentParameterAst.getValue().getRight(), is(nullValue()));
    assertThat(headersComponentParameterAst.getValue().getLeft(), equalTo("vars.headers"));

    ComponentParameterAst attachmentsComponentParameterAst = wscConsume.getParameter("Message", "attachments");
    assertThat(attachmentsComponentParameterAst, not(nullValue()));
    assertThat(attachmentsComponentParameterAst.getValue().getRight(), is(nullValue()));
    assertThat(attachmentsComponentParameterAst.getValue().getLeft(), equalTo("vars.attachments"));
  }

  @Test
  public void localErrorhandler() {
    ArtifactAst artifactAst = buildArtifactAst("parameters-test-config.xml",
                                               WebServiceConsumer.class);

    Optional<ComponentAst> optionalGroupInlineParametersShowInDsl =
        artifactAst.topLevelComponentsStream().filter(componentAst -> componentAst.getIdentifier().equals(FLOW_IDENTIFIER)
            && "errorHandlers".equals(componentAst.getComponentId().orElse(null))).findFirst();
    assertThat(optionalGroupInlineParametersShowInDsl, not(empty()));

    ComponentAst groupInlineParametersShowInDsl = optionalGroupInlineParametersShowInDsl.get();

    List<ComponentAst> children = groupInlineParametersShowInDsl.directChildrenStream().collect(toList());
    assertThat(children, hasSize(2));

    ComponentAst loggerComponent = children.get(0);
    assertThat(loggerComponent.getIdentifier().getName(), is("logger"));

    ComponentAst errorHandler = children.get(1);
    assertThat(errorHandler.getIdentifier().getName(), is("error-handler"));

    ComponentAst onErrorContinueComponent = errorHandler.directChildrenStream().findFirst()
        .orElseThrow(() -> new AssertionError("Missing on-error-continue component"));
    assertThat(onErrorContinueComponent.getIdentifier().getName(), is("on-error-continue"));

    ComponentParameterAst typeParameterAst = onErrorContinueComponent.getParameter(DEFAULT_GROUP_NAME, "type");
    assertThat(typeParameterAst, not(nullValue()));
    assertThat(typeParameterAst.getValue().getRight(), equalTo("APP:SOME"));
    assertThat(typeParameterAst.getValue().getLeft(), is(nullValue()));
  }

  @Test
  @Issue("MULE-19379")
  public void streamingStrategies() {
    // Do not run this test that verifies generationInformation if populateGenerationInformation is disabled
    assumeThat(isPopulateGenerationInformation(), is(true));

    ArtifactAst artifactAst = buildArtifactAst("parameters-test-streaming-strategy.xml",
                                               DbConnector.class, HttpConnector.class, SocketsExtension.class);

    final ComponentAst bytesStreamingOperationsFlow = artifactAst.topLevelComponentsStream()
        .filter(componentAst -> componentAst.getComponentId().map(id -> id.equals("bytesStreamingOperations")).orElse(false))
        .findFirst()
        .get();

    final ComponentAst repeatableFileStoreStream =
        (ComponentAst) bytesStreamingOperationsFlow.directChildrenStream().findFirst().get()
            .getParameter(DEFAULT_GROUP_NAME, "streamingStrategy").getValue().getRight();
    assertThat(repeatableFileStoreStream.getGenerationInformation().getSyntax().get().getElementName(),
               is("repeatable-file-store-stream"));

    final ComponentAst objectsStreamingOperations = artifactAst.topLevelComponentsStream()
        .filter(componentAst -> componentAst.getComponentId().map(id -> id.equals("objectsStreamingOperations")).orElse(false))
        .findFirst()
        .get();

    final ComponentAst repeatableFileStoreIterable =
        (ComponentAst) objectsStreamingOperations.directChildrenStream().findFirst().get()
            .getParameter(DEFAULT_GROUP_NAME, "streamingStrategy").getValue().getRight();
    assertThat(repeatableFileStoreIterable.getGenerationInformation().getSyntax().get().getElementName(),
               is("repeatable-file-store-iterable"));
  }

  @Issue("EE-7920")
  @Test
  @Story(DYNAMIC_EVALUATE)
  public void dynamicEvaluate() {
    ArtifactAst artifactAst = buildArtifactAst("dynamic-evaluate-config.xml");

    ComponentAst dynamicEvaluateFlow =
        findComponent(artifactAst.topLevelComponentsStream(), FLOW_IDENTIFIER, "dynamicEvaluate")
            .orElseThrow(() -> new AssertionError("Couldn't find 'dynamicEvaluate' flow"));

    ComponentAst dynamicEvaluate = findComponent(dynamicEvaluateFlow.directChildrenStream(), "ee:dynamic-evaluate")
        .orElseThrow(() -> new AssertionError("Couldn't find 'ee:dynamic-evaluate'"));

    ComponentParameterAst expressionParam = dynamicEvaluate.getParameter(DEFAULT_GROUP_NAME, "expression");
    assertThat(expressionParam.getValue().isLeft(), is(true));
    assertThat(expressionParam.getValue().getLeft(), is("vars.custom"));

    ComponentParameterAst parametersParam = dynamicEvaluate.getParameter(DEFAULT_GROUP_NAME, "parameters");
    assertThat(parametersParam.getValue().isLeft(), is(true));
    assertThat(parametersParam.getValue().getLeft(), is("{user: vars.param, joiner: ' and '}"));
  }

  @Test
  @Issue("MULE-19845")
  public void pojoParameterWithWrappedParamsWithSingularizableNameHasNotTheWrapperAsChild() {
    ArtifactAst artifactAst = buildArtifactAst("parameters-test-config.xml",
                                               WebServiceConsumer.class);

    final ComponentAst customEventTemplate = artifactAst.topLevelComponentsStream()
        .filter(x -> x.getIdentifier().getName().equals("custom-event-template"))
        .findFirst()
        .get();

    assertThat(customEventTemplate.directChildren(), is(Matchers.empty()));
  }

}
